/*
 * fs/sdcardfs/mmap.c
 *
 * Copyright (c) 2013 Samsung Electronics Co. Ltd
 *   Authors: Daeho Jeong, Woojoong Lee, Seunghwan Hyun,
 *               Sunghwan Yun, Sungjong Seo
 *
 * This program has been developed as a stackable file system based on
 * the WrapFS which written by
 *
 * Copyright (c) 1998-2011 Erez Zadok
 * Copyright (c) 2009     Shrikar Archak
 * Copyright (c) 2003-2011 Stony Brook University
 * Copyright (c) 2003-2011 The Research Foundation of SUNY
 *
 * This file is dual licensed.  It may be redistributed and/or modified
 * under the terms of the Apache 2.0 License OR version 2 of the GNU
 * General Public License.
 */

#include "sdcardfs.h"

static int sdcardfs_fault(struct vm_area_struct *vma, struct vm_fault *vmf)
{
	int err;
	struct file *file, *lower_file;
	const struct vm_operations_struct *lower_vm_ops;
	struct vm_area_struct lower_vma;

	memcpy(&lower_vma, vma, sizeof(struct vm_area_struct));
	file = lower_vma.vm_file;
	lower_vm_ops = SDCARDFS_F(file)->lower_vm_ops;
	BUG_ON(!lower_vm_ops);

	lower_file = sdcardfs_lower_file(file);
	/*
	 * XXX: vm_ops->fault may be called in parallel.  Because we have to
	 * resort to temporarily changing the vma->vm_file to point to the
	 * lower file, a concurrent invocation of sdcardfs_fault could see a
	 * different value.  In this workaround, we keep a different copy of
	 * the vma structure in our stack, so we never expose a different
	 * value of the vma->vm_file called to us, even temporarily.  A
	 * better fix would be to change the calling semantics of ->fault to
	 * take an explicit file pointer.
	 */
	lower_vma.vm_file = lower_file;
	err = lower_vm_ops->fault(&lower_vma, vmf);
	return err;
}

static int sdcardfs_page_mkwrite(struct vm_area_struct *vma,
								struct vm_fault *vmf)
{
	struct file *filp, *lower_filp;
	const struct vm_operations_struct *lower_vm_ops;
	struct vm_area_struct lower_vma;

	memcpy(&lower_vma, vma, sizeof(lower_vma));
	filp = lower_vma.vm_file;
	lower_vm_ops = SDCARDFS_F(filp)->lower_vm_ops;
	BUG_ON(!lower_vm_ops);

	/* FIXME: make this function not called. when lower filesystem doesn't
	 * have page_mkwrite
	 */
	if (lower_vm_ops->page_mkwrite) {
		lower_filp = sdcardfs_lower_file(filp);
		lower_vma.vm_file = lower_filp;
		return lower_vm_ops->page_mkwrite(&lower_vma, vmf);
	} else {
		return 0;
	}
}

/* XXX: to support direct I/O, a_ops->direct_IO shouldn't be null.
 */
static ssize_t sdcardfs_direct_IO(int rw, struct kiocb *iocb,
		const struct iovec *iov, loff_t offset, unsigned long nr_segs)
{
	return -ENOTSUPP;
}

/*
 * XXX: the default address_space_ops for sdcardfs is empty.  We cannot set
 * our inode->i_mapping->a_ops to NULL because too many code paths expect
 * the a_ops vector to be non-NULL.
 */
const struct address_space_operations sdcardfs_aops = {
	.direct_IO		= sdcardfs_direct_IO,
	/* empty on purpose */
};


const struct vm_operations_struct sdcardfs_vm_ops = {
	.fault			= sdcardfs_fault,
	.page_mkwrite	= sdcardfs_page_mkwrite
};
