/*
***************************************************************************
** SkFontHost_android_hy_loadfonts.cpp
***************************************************************************
**
** 2012.07.17 Android JB
**
** Mobile Communication R&D Center, Hanyang
** Sangmin, Lee (TMSword) ( Mobile Communication R&D Center / Senior Researcher )
**
** This code is a program that changes the font.
**
***************************************************************************
*/
//                                                        
#ifdef CAPP_FONTS
#include <errno.h>

#include "SkFontHost_android_hy_types.cpp"
#include "SkFontHost_android_hy_debug.cpp"
#include "SkFontHost_android_hy_api.cpp"

//                                                              
#if defined( CAPP_FONTS ) && defined( CAPP_FONTS_JP )
#include "SkFontHost_android_hy_fallback_reload.cpp"
#endif
// CAPP_FONTS_HYFONTS_JP_END

class SkFontData
{
public:
    SkFontData();
    ~SkFontData();

public:
    void init();
    void reset();
    bool checkCertify( bool system );
    bool setCertify( bool certified );
    void setFamilyAndTypeface( FamilyRec * family, FileTypeface * typeface );
    FamilyRec * getFamily();
    void deleteTypeface();
    void getFamilies( SkTDArray<FontFamily*> &families );
    void loadXml( SkTDArray<FontInitRec> &info );
    void loadFont();
    void getFullPathDownloadFont( SkString* fullpath, const char * name );

    void setFontAppName( const char * fontAppName );
    const char * getFontAppName();
    void setFontFullPath( const char * fontFullPath );
    const char * getFontFullPath();
    void setFontDatPath( const char * fontDatPath );
    const char * getFontDatPath();
    void setFontXmlPath( const char * fontXmlPath );
    const char * getFontXmlPath();
    void setWebFaceName( const char * webFaceName );
    const char * getWebFaceName();
    void setFontName( const char * fontName );
    const char * getFontName();
    void readPreData( SkStream * stream );
    void writePreData( SkFILEWStream * stream );
    void deleteFiles();

private:
    bool mCertified;
    FamilyRec * mFamily;
    FileTypeface * mTypeface;

    // Information.
    char mFontFullPath[FULLPATH_LEN];
    char mWebFaceName[WEBFACE_NAME_LEN];

    // DL font only.
    char mFontAppName[DL_FONTAPP_NAME_LEN];
    char mFontName[DL_FONT_NAME_LEN];
    char mFontDatPath[FULLPATH_LEN];
    char mFontXmlPath[FULLPATH_LEN];
};

SkFontData::SkFontData()
{
    init();
}

SkFontData::~SkFontData()
{
    //deleteTypeface();
}

void
SkFontData::init()
{
    mCertified = false;
    mFamily = NULL;
    mTypeface = NULL;

    memset( mFontFullPath, 0x00, FULLPATH_LEN );
    memset( mWebFaceName, 0x00, WEBFACE_NAME_LEN );
    memset( mFontAppName, 0x00, DL_FONTAPP_NAME_LEN );
    memset( mFontName, 0x00, DL_FONT_NAME_LEN );
    memset( mFontDatPath, 0x00, FULLPATH_LEN );
    memset( mFontXmlPath, 0x00, FULLPATH_LEN );
}

void
SkFontData::reset()
{
    init();
}

bool
SkFontData::checkCertify( bool system )
{
    SkString fullPathFontApp;
    SkStream * stFontApp = NULL;
    SkStream * stFont = NULL;
    SkStream * stDat = NULL;
    SkStream * stXml = NULL;
    skhycertified _hy_dat;

    fullPathFontApp.set( SYSTEM_DL_FONTAPP_SRC_DIR );
    fullPathFontApp.append( mFontAppName );

    if( system == true )
    {
        if( ( stFontApp = openReadStream( fullPathFontApp.c_str() ) ) == NULL )
        {
            HyLogif( "%s, not exist", fullPathFontApp.c_str() );
            goto ERROR0;
        }
        SkDELETE( stFontApp );
        //                                                                                              
        stFontApp = NULL;
    }

    if( ( stFont = openReadStream( mFontFullPath ) ) == NULL )
    {
        HyLogif( "%s, not exist", mFontFullPath  );
        goto ERROR0;
    }
    SkDELETE( stFont );
    //                                                                                              
    stFont = NULL;

    if( ( stXml = openReadStream( mFontXmlPath ) ) == NULL )
    {
        HyLogif( "%s, not exist", mFontXmlPath  );
        goto ERROR0;
    }
    SkDELETE( stXml );
    //                                                                                              
    stXml = NULL;

    if( ( stDat = openReadStream( mFontDatPath ) ) == NULL )
    {
        HyLogif( "%s, not exist", mFontDatPath  );
        goto ERROR0;
    }

    if( __certified_check( &_hy_dat, stDat, mFontFullPath ) == false )
    {
        HyLogif( "%s, verification fails", fullPathFontApp.c_str() );
        goto ERROR0;
    }

    memcpy( mFontName, _hy_dat.key7_str, ( _hy_dat.key7 + 1 ) );
    __done_dat( &_hy_dat );
    SkDELETE( stDat );

    return setCertify( true );

ERROR0:
    if( stFontApp )
        SkDELETE( stFontApp );

    if( stFont )
        SkDELETE( stFont );

    if( stDat )
        SkDELETE( stDat );

    if( stXml )
        SkDELETE( stXml );

    return setCertify( false );
}

bool
SkFontData::setCertify( bool certified )
{
    return ( mCertified = certified );
}

void
SkFontData::setFamilyAndTypeface( FamilyRec * family, FileTypeface * typeface )
{
    mFamily = family;
    mTypeface = typeface;
}

FamilyRec *
SkFontData::getFamily()
{
    return mFamily;
}

void
SkFontData::deleteTypeface()
{
    if( mTypeface )
    {
        SkDELETE( mTypeface );
    }
}

void
SkFontData::getFamilies( SkTDArray<FontFamily*> &families )
{
    parseConfigFile( mFontXmlPath, families );
}

void
SkFontData::loadXml( SkTDArray<FontInitRec> &info )
{
    SkTDArray<FontFamily*> families;
    getFamilies( families );
    info.reset();

    for( int i = 0 ; i < families.count() ; ++i )
    {
        FontFamily * family = families[i];

        for( int j = 0 ; j < family->fFileNames.count() ; ++j )
        {
            const char * filename = family->fFileNames[j];
#ifndef DEPRECATED_CODE
            HyLogif( "DEPRECATED_CODE" );
            if( haveSystemFont( filename ) )
                continue;
#endif
            FontInitRec fontInfoRecord;
            fontInfoRecord.fFileName = filename;

            if( j == 0 )
            {
                if( family->fNames.count() == 0 )
                    fontInfoRecord.fNames = ( char ** )gFBNames; // fallback.
                else
                {
                    SkTDArray<const char*> names = family->fNames;
                    const char ** nameList = ( const char ** )calloc( ( names.count() + 1 ), sizeof( char * ) );

                    if( nameList == NULL )
                    {
                        break;
                    }

                    for( int n = 0 ; n < names.count() ; ++n )
                    {
                        nameList[n] = names[n];
                    }

                    nameList[names.count()] = NULL;
                    fontInfoRecord.fNames = nameList;
                }
            }
            else
            {
                fontInfoRecord.fNames = NULL;
            }

            *info.append() = fontInfoRecord;
        }
    }

    families.deleteAll();
}

void
SkFontData::loadFont()
{
    SkTDArray<FontInitRec> info;
    SkTypeface * firstInFamily = NULL;
    loadXml( info );

    for( int i = 0 ; i < info.count() ; i++ )
    {
        const char * const * names = info[i].fNames;

        if( names != NULL )
            firstInFamily = NULL;

        bool isFixedWidth;
        SkString name;
        SkTypeface::Style style;
        bool isExpected = ( names != gFBNames );

        SkString fullpath;
        getFullPathDownloadFont( &fullpath, info[i].fFileName );

        if( !getNameAndStyleDlFont( fullpath.c_str(), &name, &style, &isFixedWidth, isExpected ) )
            continue;

        SkTypeface * tf = SkNEW_ARGS( FileTypeface, ( style, true, fullpath.c_str(), isFixedWidth ) );
        addTypefaceLocked( tf, firstInFamily );

        if( names != NULL )
        {
            firstInFamily = tf;
            FamilyRec * family = findFamilyLocked( tf );

            if( isDefaultSansFont( names ) )
            {
                FileTypeface * ftf = ( FileTypeface * )tf;
                setFamilyAndTypeface( family, ftf );
                setWebFaceName( names[0] );
            }

            while( *names )
            {
                addNameLocked( *names, family );
                names += 1;
            }
        }
    }
}

void
SkFontData::getFullPathDownloadFont( SkString* fullpath, const char * name )
{
    fullpath->set( SYSTEM_DL_FONTAPP_DST_DIR );
    fullpath->append( mFontAppName );
    fullpath->append( "/" );
    fullpath->append( name );
}

void
SkFontData::setFontAppName( const char * fontAppName )
{
    if( !fontAppName )
        return;
    strncpy( mFontAppName, fontAppName, strlen( fontAppName ) );
}

const char *
SkFontData::getFontAppName()
{
    return ( const char * )mFontAppName;
}

void
SkFontData::setFontFullPath( const char * fontFullPath )
{
    if( !fontFullPath )
        return;
    strncpy( mFontFullPath, fontFullPath, strlen( fontFullPath ) );
}

const char *
SkFontData::getFontFullPath()
{
    return ( const char * )mFontFullPath;
}

void
SkFontData::setFontDatPath( const char * fontDatPath )
{
    if( !fontDatPath )
        return;
    strncpy( mFontDatPath, fontDatPath, strlen( fontDatPath ) );
}

const char *
SkFontData::getFontDatPath()
{
    return ( const char * )mFontDatPath;
}

void
SkFontData::setFontXmlPath( const char * fontXmlPath )
{
    if( !fontXmlPath )
        return;
    strncpy( mFontXmlPath, fontXmlPath, strlen( fontXmlPath ) );
}

const char *
SkFontData::getFontXmlPath()
{
    return ( const char * )mFontXmlPath;
}

void
SkFontData::setWebFaceName( const char * webFaceName )
{
    if( !webFaceName )
        return;
    strncpy( mWebFaceName, webFaceName, strlen( webFaceName ) );
}

const char *
SkFontData::getWebFaceName()
{
    return ( const char * )mWebFaceName;
}

void
SkFontData::setFontName( const char * fontName )
{
    if( !fontName )
        return;
    memcpy( mFontName, fontName, DL_FONT_NAME_LEN );
}

const char *
SkFontData::getFontName()
{
    return ( const char * )mFontName;
}

void
SkFontData::readPreData( SkStream * stream )
{
    stream->read( mFontAppName, DL_FONTAPP_NAME_LEN );
    stream->read( mFontFullPath, FULLPATH_LEN );
    stream->read( mFontDatPath, FULLPATH_LEN );
    stream->read( mFontXmlPath, FULLPATH_LEN );
    stream->read( mFontName, DL_FONT_NAME_LEN );
    stream->read( mWebFaceName, WEBFACE_NAME_LEN );
}

void
SkFontData::writePreData( SkFILEWStream * stream )
{
    stream->write( mFontAppName, DL_FONTAPP_NAME_LEN );
    stream->write( mFontFullPath, FULLPATH_LEN );
    stream->write( mFontDatPath, FULLPATH_LEN );
    stream->write( mFontXmlPath, FULLPATH_LEN );
    stream->write( mFontName, DL_FONT_NAME_LEN );
    stream->write( mWebFaceName, WEBFACE_NAME_LEN );
}

void
SkFontData::deleteFiles()
{
    SkString delCmdFiles;
    delCmdFiles.set( "rm " );
    delCmdFiles.append( SYSTEM_DL_FONTAPP_DST_DIR );
    delCmdFiles.append( mFontAppName );
    delCmdFiles.append( "/*" );
    system( delCmdFiles.c_str() );

    SkString delCmdDir;
    delCmdDir.set( "rmdir " );
    delCmdDir.append( SYSTEM_DL_FONTAPP_DST_DIR );
    delCmdDir.append( mFontAppName );
    system( delCmdDir.c_str() );
 }

class SkFontDataList
{
public:
    SkFontDataList();
    ~SkFontDataList();

public:
    void init();
    bool isMax();
    bool setMaxFonts( uint32_t max );
    void resetFonts();
    void release();
    uint32_t numFonts();
    bool addFonts( SkFontData * font );
    FamilyRec * getFamily( uint32_t idxFont );
    SkFontData ** getFonts();
    SkFontData * getFont( uint32_t fontIndex );

private:
    uint32_t mMaxFonts;
    uint32_t mNumFonts;
    SkFontData ** mFonts;
};

SkFontDataList::SkFontDataList()
{
    init();
}

SkFontDataList::~SkFontDataList()
{
    release();
}

void
SkFontDataList::init()
{
    mMaxFonts = 0;
    mNumFonts = 0;
    mFonts = NULL;
}

bool
SkFontDataList::isMax()
{
    return ( ( mMaxFonts > 0 ) ? true : false );
}

bool
SkFontDataList::setMaxFonts( uint32_t maxFonts )
{
    if( maxFonts > 0 )
    {
        if( ( mFonts = ( SkFontData ** )calloc( maxFonts, sizeof( SkFontData * ) ) ) == NULL )
        {
            HyLogef( "Fonts array, alloc()" );
            goto ERROR0;
        }
        mMaxFonts = maxFonts;
        return true;
    }
ERROR0:
    return false;
}

// Clear list.
void
SkFontDataList::resetFonts()
{
    if( isMax() )
    {
        for( uint32_t n = 0 ; n < mNumFonts ; n++ )
        {
            if( mFonts[n] )
            {
                SkDELETE( mFonts[n] );
                mFonts[n] = NULL;
            }
        }
        mNumFonts = 0;
    }
}

// Delete list.
void
SkFontDataList::release()
{
    if( isMax() )
    {
        resetFonts();
        free( mFonts );
        mFonts = NULL;
    }
}

uint32_t
SkFontDataList::numFonts()
{
    return mNumFonts;
}

bool
SkFontDataList::addFonts( SkFontData * font )
{
    if( ( mNumFonts + 1 ) < mMaxFonts )
    {
        mFonts[mNumFonts++] = font;
        return true;
    }
    return false;
}

FamilyRec *
SkFontDataList::getFamily( uint32_t idxFont )
{
    if( idxFont < mNumFonts )
    {
        SkFontData * font = mFonts[idxFont];

        if( font )
            return font->getFamily();
    }
    return NULL;
}

SkFontData **
SkFontDataList::getFonts()
{
    return mFonts;
}

SkFontData *
SkFontDataList::getFont( uint32_t fontIndex )
{
    return mFonts[fontIndex];
}

// The default font list.
class SkFontEmbeddedList : public SkFontDataList
{
public:
    SkFontEmbeddedList();
    ~SkFontEmbeddedList();

public:
    void initFontsLocked();
};

SkFontEmbeddedList::SkFontEmbeddedList()
{
    setMaxFonts( SYSTEM_EMBEDDED_FAMILY_MAX );
}

SkFontEmbeddedList::~SkFontEmbeddedList()
{

}

void
SkFontEmbeddedList::initFontsLocked()
{
    if( gDefaultNormal ) return;
    SkASSERT( gUniqueFontID == 0 );
    loadFontInfoLocked();
    gFallbackFonts.reset();

    SkTypeface * firstInFamily = NULL;

    for( int i = 0 ; i < gSystemFonts.count() ; i++ )
    {
        const char * const * names = gSystemFonts[i].fNames;

        if( names != NULL )
            firstInFamily = NULL;

        bool isFixedWidth;
        SkString name;
        SkTypeface::Style style;
        bool isExpected = ( names != gFBNames );

        if( !getNameAndStyle( gSystemFonts[i].fFileName, &name, &style, &isFixedWidth, isExpected ) )
        {
            sk_atomic_inc( &gUniqueFontID );
            continue;
        }

        SkString fullpath;
        SkTypeface * tf = NULL;

        getFullPathForSysFonts( &fullpath, gSystemFonts[i].fFileName );
        tf = SkNEW_ARGS( FileTypeface, ( style, true, fullpath.c_str(), isFixedWidth ) );
        addTypefaceLocked( tf, firstInFamily );

        if( names != NULL )
        {
            if( names == gFBNames )
                *gFallbackFonts.append() = tf->uniqueID();

            firstInFamily = tf;
            FamilyRec * family = findFamilyLocked( tf );

            if( isDefaultSansFont( names ) == true )
            {
                FileTypeface * ftf = ( FileTypeface * )tf;
                SkFontData * font = SkNEW( SkFontData );

                if( font )
                {
                    font->setFamilyAndTypeface( family, ftf );
                    font->setFontFullPath( ftf->getFilePath() );
                    font->setWebFaceName( names[0] );
                    addFonts( font );
                }
                else
                {
                    HyLogef( "SkNEW( SkFontData ) (%s)", ftf->getFilePath() );
                    continue;
                }
            }

            while( *names )
            {
                addNameLocked( *names, family );
                names += 1;
            }
        }
    }
}

// Verified list of fonts.
class SkFontDownloadList : public SkFontDataList
{
public:
    SkFontDownloadList();
    ~SkFontDownloadList();

public:
    bool isExist( const char * fontAppName );
};

SkFontDownloadList::SkFontDownloadList()
{
    setMaxFonts( SYSTEM_DOWNLOAD_FAMILY_MAX );
}

SkFontDownloadList::~SkFontDownloadList()
{

}

bool
SkFontDownloadList::isExist( const char * fontAppName )
{
    if( numFonts() == 0 )
        return false;
    else
    {
        SkFontData ** fonts = getFonts();

        for( uint32_t n = 0 ; n < numFonts() ; n++ )
            if( !strcmp( fontAppName, fonts[n]->getFontAppName() ) )
                return true;

        return false;
    }
}

// Search the list of fonts.
class SkFontSearchList : public SkFontDataList
{
public:
    SkFontSearchList();
    ~SkFontSearchList();

public:
    void setError( uint32_t error );
    uint32_t getError();
    SearchState setState( SearchState state );
    SearchState getState();

    void ready();
    SearchState search( SkFontDownloadList * dlFonts );

public:
    uint32_t mError;
    SearchState mState;
};

SkFontSearchList::SkFontSearchList()
{
    setMaxFonts( SYSTEM_SEARCH_FAMILY_MAX );
    setError( ERR_NO );
    setState( STATE_DL_FONTAPP_KEEP );
}

SkFontSearchList::~SkFontSearchList()
{

}

void
SkFontSearchList::setError( uint32_t error )
{
    mError = error;
}

uint32_t
SkFontSearchList::getError()
{
    return mError;
}

SearchState
SkFontSearchList::setState( SearchState state )
{
    return ( mState = state );
}

SearchState
SkFontSearchList::getState()
{
    return mState;
}

void
SkFontSearchList::ready()
{
    // Clear list.
    resetFonts();
    setError( ERR_NO );
    setState( STATE_DL_FONTAPP_KEEP );
}

SearchState
SkFontSearchList::search( SkFontDownloadList * dlFonts )
{
    DIR * dp = NULL;
    struct dirent * d = NULL;

    ready();

    if( ( dp = opendir( SYSTEM_DL_FONTAPP_SRC_DIR ) ) == NULL )
    {
        HyLogef( "%s, opendir()", SYSTEM_DL_FONTAPP_SRC_DIR );
        goto ERROR0;
    }

    while( ( d = readdir( dp ) ) != NULL )
    {
        if( !strncmp( d->d_name, ".", 1 ) )
            continue;

        if( !strncmp( d->d_name, DL_FONTAPP_SEARCH_STR, DL_FONTAPP_SEARCH_LEN ) )
        {
            if( dlFonts->isExist( d->d_name ) == true )
                continue;
            else
            {
                SkFontData * font = SkNEW( SkFontData );

                if( font )
                {
                    SkString fontFullPath;
                    SkString fontDatPath;
                    SkString fontXmlPath;

                    font->setFontAppName( d->d_name );

                    fontFullPath.set( SYSTEM_DL_FONTAPP_DST_DIR );
                    fontFullPath.append( font->getFontAppName() );
                    fontFullPath.append( DLFILE_TTF );

                    fontDatPath.set( SYSTEM_DL_FONTAPP_DST_DIR );
                    fontDatPath.append( font->getFontAppName() );
                    fontDatPath.append( DLFILE_DAT );

                    fontXmlPath.set( SYSTEM_DL_FONTAPP_DST_DIR );
                    fontXmlPath.append( font->getFontAppName() );
                    fontXmlPath.append( DLFILE_XML );

                    font->setFontFullPath( fontFullPath.c_str() );
                    font->setFontDatPath( fontDatPath.c_str() );
                    font->setFontXmlPath( fontXmlPath.c_str() );
                    addFonts( font );
                }
                else
                {
                    HyLogef( "Error : SkNEW( SkFontData )." );
                    continue;
                }
            }
        }
    }

    closedir( dp );

    if( numFonts() > 0 )
        return setState( STATE_DL_FONTAPP_FIND );
    else
        return setState( STATE_DL_FONTAPP_KEEP );

ERROR0:
    setError( ERR_YES );
    return setState( STATE_DL_FONTAPP_KEEP );
}

class SkFontManager
{
public:
    SkFontManager();
    ~SkFontManager();

public:
    SearchState searchFontApps();
    void initSystemFontsLocked();
    void initDownloadFontsLocked();
    uint32_t getCurrentFontIndex();
    void saveFontIndex( uint32_t fontIndex );
    uint32_t getFamilyCount();
    uint32_t getEmbeddedFamilyCount();
    LoadState loadPreData( uint32_t * newFontIndex, bool boot );
    bool openPreDataRead( SkStream ** preData );
    bool savePreData();

    uint32_t getUpdateVersion();
    void setUpdateVersion( uint32_t updateVersion );
    void updateDefaultNormal();
    FamilyRec * getDefaultFamily();
    FamilyRec * getFamily( uint32_t fontIndex );
    SkFontData * getFont( uint32_t fontIndex );
    SkFontData * getFont();

    const char * getDownloadFontName( uint32_t fontIndex );
    const char * getFontWebFaceName( uint32_t fontIndex );
    const char * getFontFullPath( uint32_t fontIndex );

    uint32_t getNumNewFonts();
    const char * getNewFontAppName( uint32_t fontIndex );
    SearchState getCheckNewFont( uint32_t fontIndex );
    bool addNewFont( uint32_t fontIndex );
    bool copyFont( SkFontData * dst, SkFontData * src );
    void setPermission( SkFontData * src );
    void saveNewFont();

    LoadState updateFontManager();
    LoadState checkLoadedFonts();
    LoadState checkPreData();

    //                                                                             
    #if defined( CAPP_FONTS_JP )
        void updateJapanFallbackFonts();
    #endif
    // CAPP_FONTS_HYFONTS_JP_END

public:
    static bool createFontManager( SkFontManager ** fontManager );

private:
    SkFontEmbeddedList mEmbeddedFonts;
    SkFontDownloadList mDownloadFonts;
    SkFontSearchList mSearchFonts;

    // PreData.
    preDataHeader mHeader;
    uint32_t mFontIndex;
};

SkFontManager::SkFontManager()
{
    memset( &mHeader, 0x00, sizeof( preDataHeader ) );
    mFontIndex = SYSTEM_DEFAULT_FAMILY_INDEX;
}

SkFontManager::~SkFontManager()
{

}

SearchState
SkFontManager::searchFontApps()
{
    return mSearchFonts.search( &mDownloadFonts );
}

void
SkFontManager::initSystemFontsLocked()
{
    mEmbeddedFonts.initFontsLocked();
    initDownloadFontsLocked();
    updateDefaultNormal();
    //                                                                             
    #if defined( CAPP_FONTS_JP )
        updateJapanFallbackFonts();
    #endif
    // CAPP_FONTS_HYFONTS_JP_END
    HyLogi( "Number of families : %d ( BootTypeface(%d) : %s )", getFamilyCount(), mFontIndex, getFontWebFaceName( mFontIndex ) );
}

void
SkFontManager::initDownloadFontsLocked()
{
    uint32_t newFontIndex = 0;
    if( loadPreData( &newFontIndex, true ) >= load_change )
    {
        saveFontIndex( newFontIndex );
        savePreData();
        HyLogi( "Download font information has changed." );
    }
}

uint32_t
SkFontManager::getCurrentFontIndex()
{
#ifdef FONTCONFIG_DAT_PATCH
    char wfaceName[WEBFACE_NAME_LEN];
    memset( wfaceName, 0x00, WEBFACE_NAME_LEN );
    uint32_t fontIndex = readFontConfigV2( wfaceName );
    if( fontIndex >= getFamilyCount() )
        fontIndex = SYSTEM_DEFAULT_FAMILY_INDEX;
    else
    {
        if( strnlen( wfaceName, WEBFACE_NAME_LEN ) > 0 )
        {
            if( strncmp( wfaceName, getFontWebFaceName( fontIndex ), WEBFACE_NAME_LEN ) != 0 )
            {
                uint32_t numAllFonts = getFamilyCount();
                for( uint32_t n = 0 ; n < numAllFonts ; n++ )
                {
                    if( strncmp( wfaceName, getFontWebFaceName( n ), WEBFACE_NAME_LEN ) == 0 )
                    {
                        fontIndex = n;
                        goto END0;
                    }
                }
                fontIndex = SYSTEM_DEFAULT_FAMILY_INDEX;
                goto END0;
            }
        }
    }
END0:
    return ( mFontIndex = fontIndex );
#else
    uint32_t fontIndex = readFontConfig();
    if( fontIndex >= getFamilyCount() )
        fontIndex = SYSTEM_DEFAULT_FAMILY_INDEX;
    return ( mFontIndex = fontIndex );
#endif
}

void
SkFontManager::saveFontIndex( uint32_t fontIndex )
{
#ifdef FONTCONFIG_DAT_PATCH
    if( ( fontIndex == mFontIndex ) || ( fontIndex >= getFamilyCount() ) )
        return;
    saveFontConfigV2( fontIndex, getFontWebFaceName( fontIndex ) );
#else
    if( ( fontIndex == mFontIndex ) || ( fontIndex >= getFamilyCount() ) )
        return;
    saveFontConfig( fontIndex );
#endif
}

uint32_t
SkFontManager::getFamilyCount()
{
    return ( mEmbeddedFonts.numFonts() + mDownloadFonts.numFonts() );
}

uint32_t
SkFontManager::getEmbeddedFamilyCount()
{
    return mEmbeddedFonts.numFonts();
}

LoadState
SkFontManager::loadPreData( uint32_t * newFontIndex, bool boot )
{
    LoadState state = load_true;
    preDataHeader header = { 0, 0, 0 };
    uint32_t curFontIndex = ( boot == true ? readFontConfig() : getCurrentFontIndex() );
    uint32_t curNewFontIndex = curFontIndex;
    uint32_t numEmbeddedFonts = mEmbeddedFonts.numFonts();
    bool statePass = ( curFontIndex < numEmbeddedFonts ? true : false );
    bool system = ( newFontIndex != NULL ? true : false );
    SkStream * stream = NULL;

    if( openPreDataRead( &stream ) == false )
    {
        if( no_such_file() || permission_denied() )
            goto SKIP_ERROR0;

        HyLogef( "openPreDataRead(%s (%d))", strerror( errno ), errno );
        goto ERROR0;
    }

    stream->read( &header, SYSTEM_DL_PREDATA_HEADER_LEN );

    if( header.tag != __key() )
    {
        HyLogef( "Wrong tag" );
        goto ERROR0;
    }

LOAD_START:
    if( getUpdateVersion() == 0 ) // Boot Load & reload.
    {
        setUpdateVersion( header.updateVersion );

        if( header.count == 0 )
            goto SUCCEED0;

        for( uint32_t n = 0 ; n < header.count ; n++ )
        {
            SkFontData * font = SkNEW( SkFontData );

            if( font )
            {
                font->readPreData( stream );

                if( font->checkCertify( system ) == true )
                {
                    font->loadFont();
                    mDownloadFonts.addFonts( font );
                }
                else
                {
                    if( ( system == true ) && ( statePass == false ) )
                    {
                        if( state != load_restart )
                        {
                            if( ( numEmbeddedFonts + n ) == curFontIndex )
                            {
                                state = load_restart;
                            }
                            else
                            {
                                state = load_change;
                                if( ( numEmbeddedFonts + n ) < curFontIndex )
                                    curNewFontIndex--;
                            }
                        }
                    }
                    else
                    {
                        state = load_change;
                    }

                    font->deleteFiles();
                    SkDELETE( font );
                    font = NULL;
                }
            }
            else
            {
                HyLogef( "SkNEW( SkFontData )" );
                continue;
            }
        }
    }
    else // Fonts change
    {
        if( getUpdateVersion() == header.updateVersion )
        {
            goto SUCCEED0;
        }
        else
        {
            mDownloadFonts.resetFonts();
            setUpdateVersion( 0 );
            goto LOAD_START;
        }
    }

SUCCEED0:
    if( ( system == true ) && ( statePass == false ) )
    {
        switch( state )
        {
        case load_change: *newFontIndex = curNewFontIndex; break;
        case load_restart: *newFontIndex = SYSTEM_DEFAULT_FAMILY_INDEX; break;
        default: *newFontIndex = curFontIndex; break;
        }
    }
    else if( ( system == true ) && ( statePass == true ) )
    {
        *newFontIndex = curFontIndex;
    }

    if( stream )
        SkDELETE( stream );
    return state;

SKIP_ERROR0:
    return load_false;

ERROR0:
    if( stream )
        SkDELETE( stream );
    return load_false;
}

bool
SkFontManager::openPreDataRead( SkStream ** preData )
{
    SkStream * stream = openReadStream( SYSTEM_DL_PREDATA_FILE );

    if( !stream )
    {
        if( permission_denied() )
            goto SKIP_ERROR0;

        if( !no_such_file() )
            goto ERROR0;

        goto SKIP_ERROR0;
    }

    *preData = stream;
    return true;

SKIP_ERROR0:
    return false;

ERROR0:
    if( stream )
        SkDELETE( stream );
    return false;
}

LoadState
SkFontManager::checkPreData()
{
    LoadState state = load_true;
    SkStream * inStream = openReadStream( SYSTEM_DL_PREDATA_FILE );
    preDataHeader header = { 0, 0, 0 };

    if( !inStream )
    {
        state = load_false;
        goto END0;
    }

    inStream->read( &header, SYSTEM_DL_PREDATA_HEADER_LEN );

    if( header.tag != __key() )
    {
        state = load_false;
        goto END0;
    }

    if( getUpdateVersion() == header.updateVersion )
    {
        state = load_true;
        goto END0;
    }
    else
    {
        state = load_change;
        goto END0;
    }

END0:
    if( inStream )
        SkDELETE( inStream );
    return state;
}


bool
SkFontManager::savePreData()
{
    SkStream * inStream = openReadStream( SYSTEM_DL_PREDATA_FILE );
    SkFILEWStream * outStream = NULL;
    preDataHeader header = { 1, __key(), 0 };
    SkString permission;

    if( !inStream )
    {
        //                                                                              
        setUpdateVersion( 2 );
        // CAPP_FONTS_HYFONTS_END
        header.updateVersion++;
        header.count = mDownloadFonts.numFonts();
        goto SAVE0;
    }
    else
    {
        inStream->read( &header, SYSTEM_DL_PREDATA_HEADER_LEN );

        if( header.tag != __key() )
        {
            header.updateVersion = 1;
            header.tag = __key();
            header.count = mDownloadFonts.numFonts();
            goto SAVE1;
        }

        //                                                                                                      
        if( getUpdateVersion() == 1 && header.updateVersion == 1 )
            setUpdateVersion( 2 );

        header.count = mDownloadFonts.numFonts();
        header.updateVersion++;
        if( header.updateVersion >= 0xFFFFFFFE )
            header.updateVersion = 2;
    }

SAVE1:
    SkDELETE( inStream );
    inStream = NULL;

SAVE0:
    outStream = openWriteStream( SYSTEM_DL_PREDATA_FILE );

    if( !outStream )
    {
        if( permission_denied() )
            goto SKIP_ERROR0;
        HyLogef( "outStream is null" );
        goto ERROR0;
    }

    outStream->write( &header, SYSTEM_DL_PREDATA_HEADER_LEN );

    for( uint32_t n = 0 ; n < header.count ; n++ )
    {
        SkFontData * font = mDownloadFonts.getFont( n );
        font->writePreData( outStream );
    }

    SkDELETE( outStream );
    outStream = NULL;

    //setUpdateVersion( header.updateVersion );

    permission.set( SYSTEM_FONT_PERMISSION_SET );
    permission.append( SYSTEM_DL_PREDATA_FILE );
    system( permission.c_str() );

    HyLogi( "Font Configuration Data (%d), saved.", header.updateVersion );
    return true;

SKIP_ERROR0:
    return false;

ERROR0:
    if( outStream )
        SkDELETE( outStream );
    return false;
}

uint32_t
SkFontManager::getUpdateVersion()
{
    return mHeader.updateVersion;
}

void
SkFontManager::setUpdateVersion( uint32_t updateVersion )
{
    mHeader.updateVersion = updateVersion;
}

void
SkFontManager::updateDefaultNormal()
{
    gDefaultFamily = getDefaultFamily();
    gDefaultNormal = findBestFaceLocked( gDefaultFamily, SkTypeface::kNormal );
}

//                                                                             
#if defined( CAPP_FONTS_JP )
void
SkFontManager::updateJapanFallbackFonts()
{
    reloadJapanFallbackFontsLocked( getFont()->getWebFaceName() );
}
#endif
// CAPP_FONTS_HYFONTS_JP_END

FamilyRec *
SkFontManager::getDefaultFamily()
{
    uint32_t fontIndex = getCurrentFontIndex();
    return getFamily( fontIndex );
}

FamilyRec *
SkFontManager::getFamily( uint32_t fontIndex )
{
    if( fontIndex >= mEmbeddedFonts.numFonts() )
        return mDownloadFonts.getFamily( fontIndex - mEmbeddedFonts.numFonts() );
    return mEmbeddedFonts.getFamily( fontIndex );
}

SkFontData *
SkFontManager::getFont( uint32_t fontIndex )
{
    if( fontIndex >= mEmbeddedFonts.numFonts() )
        return mDownloadFonts.getFont( fontIndex - mEmbeddedFonts.numFonts() );
    return mEmbeddedFonts.getFont( fontIndex );
}

SkFontData *
SkFontManager::getFont()
{
    return getFont( mFontIndex );
}

const char *
SkFontManager::getDownloadFontName( uint32_t fontIndex )
{
    return mDownloadFonts.getFont( fontIndex - mEmbeddedFonts.numFonts() )->getFontName();
}

const char *
SkFontManager::getFontWebFaceName( uint32_t fontIndex )
{
    return getFont( fontIndex )->getWebFaceName();
}

const char *
SkFontManager::getFontFullPath( uint32_t fontIndex )
{
    return getFont( fontIndex )->getFontFullPath();
}

uint32_t
SkFontManager::getNumNewFonts()
{
    return mSearchFonts.numFonts();
}

const char *
SkFontManager::getNewFontAppName( uint32_t fontIndex )
{
    if( fontIndex >= mSearchFonts.numFonts() )
        return NULL;
    return mSearchFonts.getFont( fontIndex )->getFontAppName();
}

SearchState
SkFontManager::getCheckNewFont( uint32_t fontIndex )
{
    if( fontIndex >= mSearchFonts.numFonts() )
        return STATE_DL_FONTAPP_KEEP;

    if( mSearchFonts.getFont( fontIndex )->checkCertify( true ) == true )
        return STATE_DL_FONTAPP_UPDATE;
    //                                                                                              
    else
        mSearchFonts.getFont( fontIndex )->deleteFiles();

    return STATE_DL_FONTAPP_KEEP;
}

bool
SkFontManager::addNewFont( uint32_t fontIndex )
{
    if( fontIndex >= mSearchFonts.numFonts() )
        goto ERROR0;
    else
    {
        SkFontData * newFont = SkNEW( SkFontData );

        if( newFont )
        {
            if( copyFont( newFont, mSearchFonts.getFont( fontIndex ) ) == false )
                goto ERROR1;

            if( mDownloadFonts.addFonts( newFont ) == false )
                goto ERROR1;

            setPermission( newFont );
            //                                                                                            
            newFont->loadFont();
            return true;
        }

    ERROR1:
        if( newFont )
            SkDELETE( newFont );
        goto ERROR0;
    }

ERROR0:
    return false;
}

bool
SkFontManager::copyFont( SkFontData * dst, SkFontData * src )
{
    dst->setFontFullPath( src->getFontFullPath() );
    dst->setWebFaceName( src->getWebFaceName() );
    dst->setFontAppName( src->getFontAppName() );
    dst->setFontName( src->getFontName() );
    dst->setFontDatPath( src->getFontDatPath() );
    dst->setFontXmlPath( src->getFontXmlPath() );
    return dst->checkCertify( true );
}

void
SkFontManager::setPermission( SkFontData * src )
{
    SkString path;
    SkString file;

    path.set( SYSTEM_FONT_PERMISSION_SET );
    path.append( SYSTEM_DL_FONTAPP_DST_DIR );
    path.append( src->getFontAppName() );

    file.set( SYSTEM_FONT_PERMISSION_SET );
    file.append( SYSTEM_DL_FONTAPP_DST_DIR );
    file.append( src->getFontAppName() );
    file.append( "/*" );

    system( path.c_str() );
    system( file.c_str() );
}

void
SkFontManager::saveNewFont()
{
    savePreData();
}

LoadState
SkFontManager::updateFontManager()
{
    LoadState state = load_true;
    uint32_t newFontIndex = 0;

    if( ( ( state = checkLoadedFonts() ) == load_change  ) || ( ( state = checkPreData() ) == load_change ) )
    {
        mHeader.updateVersion--;
        state = loadPreData( &newFontIndex, false );
    }

    switch( state )
    {
    case load_change:
    case load_restart:
        savePreData();
        saveFontIndex( newFontIndex );
        return state;
    default:
        return load_true;
    }
}

LoadState
SkFontManager::checkLoadedFonts()
{
    uint32_t numFonts = mDownloadFonts.numFonts();

    for( uint32_t n = 0 ; n < numFonts ; n++ )
    {
        SkFontData * font = mDownloadFonts.getFont( n );
        if( font->checkCertify( true ) == false )
            return load_change;
    }
    return load_true;
}

// Static functions.
bool
SkFontManager::createFontManager( SkFontManager ** fontManager )
{
    SkFontManager * fntMgr = SkNEW( SkFontManager );

    if( fntMgr )
    {
        *fontManager = fntMgr;
        return true;
    }
    *fontManager = NULL;
    HyLogef( "SkNEW( SkFontManager )" );
    return false;
}

// FontManager Instance.
static SkFontManager * gFontManager = NULL;

static bool
createFontManager()
{
    if( SkFontManager::createFontManager( &gFontManager ) == false )
    {
        HyLogef( "Font Manager creation failed." );
        return false;
    }
    HyLogi( "Font Manager creation successful." );
    return true;
}

static SkFontManager *
getInstance()
{
    return gFontManager;
}

static bool
initSystemFontsLocked()
{
    if( !createFontManager() )
    {
        return false;
    }
    getInstance()->initSystemFontsLocked();
    SkDEBUGCODE( dumpGlobalsLocked() );
    return true;
}

static void
loadSystemFontsLocked()
{
#if !defined(SK_BUILD_FOR_ANDROID_NDK)
    static char prevLanguage[3];
    static char prevRegion[3];
    char language[3] = "";
    char region[3] = "";

    getLocale( language, region );

    if( !gDefaultNormal )
    {
        HyLogi( "Start to preload the system font." );
        strncpy( prevLanguage, language, 2 );
        strncpy( prevRegion, region, 2 );
        initSystemFontsLocked();
    }
    else if( strncmp( language, prevLanguage, 2 ) || strncmp( region, prevRegion, 2 ) )
    {
        strncpy( prevLanguage, language, 2 );
        strncpy( prevRegion, region, 2 );
//                                                              
#ifndef CAPP_FONTS_JP
        reloadFallbackFontsLocked();
#endif
// CAPP_FONTS_HYFONTS_JP_END
    }
#else
    if( !gDefaultNormal )
    {
        initSystemFontsLocked();
        reloadFallbackFontsLocked();
    }
#endif
}

static SkTypeface *
findSansSerifFaceLocked( const char familyName[], SkTypeface::Style style )
{
    const char * facename = getInstance()->getFont()->getWebFaceName();
    if( !strncmp( "roboto", facename, strlen( facename ) ) )
    {
        return findTypefaceLocked( familyName, style );
    }
    else
    {
        return findBestFaceLocked( gDefaultFamily, style );
    }
}

static SkTypeface *
createTypefaceLocked( const SkTypeface * familyFace, const char familyName[], const void * data, size_t bytelength, SkTypeface::Style style )
{
    loadSystemFontsLocked();
    style = ( SkTypeface::Style )( style & SkTypeface::kBoldItalic );
    SkTypeface * tf = NULL;

    if( NULL != familyFace )
    {
         tf = findTypefaceLocked( familyFace, style );
    }
    else if( NULL != familyName )
    {
        if( compareSystemFamilyName( familyName ) == true )
        {
            tf = findBestFaceLocked( gDefaultFamily, style );
        }
#ifndef DEPRECATED_CODE
        else if( compareSansSerifFamilyName( familyName ) == true )
        {
            HyLogif( "DEPRECATED_CODE" );
            tf = findSansSerifFaceLocked( familyName, style );
        }
#endif
        else
        {
            tf = findTypefaceLocked( familyName, style );
        }
    }

    if( NULL == tf )
    {
        tf = findBestFaceLocked( gDefaultFamily, style );
    }

    tf->ref();
    return tf;
}

//                                                                                  
void SkFontHost::Serialize(const SkTypeface* face, SkWStream* stream) {
    // lookup and record if the font is custom (i.e. not a system font)
    bool isCustomFont = !((FamilyTypeface*)face)->isSysFont();
    stream->writeBool(isCustomFont);

    if (isCustomFont) {
        SkStream* fontStream = ((FamilyTypeface*)face)->openStream();

        // store the length of the custom font
        uint32_t len = fontStream->getLength();
        stream->write32(len);

        // store the entire font in the serialized stream
        void* fontData = malloc(len);

        fontStream->read(fontData, len);
        stream->write(fontData, len);

        fontStream->unref();
        free(fontData);
//      SkDebugf("--- fonthost custom serialize %d %d\n", face->style(), len);

    } else {
        const char* name = ((FamilyTypeface*)face)->getFilePath(); // const char* name = ((FamilyTypeface*)face)->getUniqueString();

        stream->write8((uint8_t)face->style());

        if (NULL == name || 0 == *name) {
            stream->writePackedUInt(0);
//          SkDebugf("--- fonthost serialize null\n");
        } else {
            uint32_t len = strlen(name);
            stream->writePackedUInt(len);
            stream->write(name, len);
//          SkDebugf("--- fonthost serialize <%s> %d\n", name, face->style());
        }
    }
}

SkTypeface* SkFontHost::Deserialize(SkStream* stream) {
    SkAutoMutexAcquire  ac(gFamilyHeadAndNameListMutex);
    return deserializeLocked(stream);
}

static SkTypeface* deserializeLocked(SkStream* stream) {
    loadSystemFontsLocked();

    // check if the font is a custom or system font
    bool isCustomFont = stream->readBool();

    if (isCustomFont) {

        // read the length of the custom font from the stream
        uint32_t len = stream->readU32();

        // generate a new stream to store the custom typeface
        SkMemoryStream* fontStream = new SkMemoryStream(len);
        stream->read((void*)fontStream->getMemoryBase(), len);

        SkTypeface* face = createTypefaceFromStreamLocked(fontStream);

        fontStream->unref();

//      SkDebugf("--- fonthost custom deserialize %d %d\n", face->style(), len);
        return face;

    } else {
        int style = stream->readU8();

        int len = stream->readPackedUInt();
        if (len > 0) {
            SkString str;
            str.resize(len);
            stream->read(str.writable_str(), len);

            // Embedded fonts.
            for (int i = 0; i < gSystemFonts.count(); i++) {
                SkString fullpath;
                getFullPathForSysFonts( &fullpath, gSystemFonts[i].fFileName );
                const char * name = fullpath.c_str();
                if (strcmp(name, str.c_str()) == 0) {
                    // backup until we hit the fNames
                    for (int j = i; j >= 0; --j) {
                        if (gSystemFonts[j].fNames != NULL) {
                            return createTypefaceLocked(NULL,
                                    gSystemFonts[j].fNames[0], NULL, 0,
                                    (SkTypeface::Style)style);
                        }
                    }
                }
            }

            // Download fonts.
            SkFontManager * fmg = getInstance();
            uint32_t numAllFonts = fmg->getFamilyCount();
            uint32_t numEmbeddedFonts = fmg->getEmbeddedFamilyCount();
            for (int i = numEmbeddedFonts; i < numAllFonts; i++) {
                SkFontData * font = fmg->getFont( i );
                const char * name = font->getFontFullPath();
                if (strcmp(name, str.c_str()) == 0) {
                    // backup until we hit the fNames
                    return createTypefaceLocked(NULL,
                               font->getWebFaceName(), NULL, 0,
                               (SkTypeface::Style)style);
                }
            }
            //CAPP_FONTS_HYFONTS [20130318] To fix crash in Browser when loading saved page
            return createTypefaceLocked(NULL, gSystemFonts[0].fNames[0], NULL, 0, (SkTypeface::Style)style);
        }
    }
    return NULL;
}

#include "SkFontHost_android_hy_fonthost.cpp"
#endif
// CAPP_FONTS_HYFONTS_END
