/* lge_ts_core.c
 *
 * Copyright (C) 2011 LGE.
 *
 * Author: yehan.ahn@lge.com, hyesung.shin@lge.com
 * Modified : WX-BSP-TS@lge.com
 *
 * This software is licensed under the terms of the GNU General Public
 * License version 2, as published by the Free Software Foundation, and
 * may be copied, distributed, and modified under those terms.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

#define LGE_TS_CORE_C

#include <asm/atomic.h>

#include <linux/version.h>
#include <linux/types.h>
#include <linux/err.h>
#include <linux/moduleparam.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/hrtimer.h>
#include <linux/time.h>
#include <linux/jiffies.h>
#include <linux/slab.h>
#include <linux/device.h>
#include <linux/gpio.h>
#include <linux/i2c.h>
#include <linux/input.h>
#include <linux/input/mt.h>
#include <linux/firmware.h>
#ifdef CONFIG_OF
#include <linux/of.h>
#include <linux/of_gpio.h>
#endif
#if defined(CONFIG_FB)
#include <linux/fb.h>
#include <linux/notifier.h>
#endif
#if defined(CONFIG_HAS_EARLYSUSPEND)
#include <linux/earlysuspend.h>
#endif
#include <linux/completion.h>

#include "lge_ts_core.h"

#define LGE_TOUCH_NAME		"lge_touch"

struct lge_touch_data {
	void				*h_touch;
	atomic_t			device_init;
	u8				work_sync_err_cnt;
	u8				ic_init_err_cnt;
	u8				ic_error_cnt;
	volatile int			curr_pwr_state;

	struct i2c_client 		*client;
	struct input_dev 		*input_dev;
	struct hrtimer 			timer;
	struct hrtimer 			trigger_timer;
	struct work_struct  		work;
	struct delayed_work		work_init;
	struct delayed_work		work_touch_lock;
	struct work_struct  		work_fw_upgrade;
#if defined(CONFIG_FB)
	struct notifier_block		fb_notifier_block;
#endif
#if defined(CONFIG_HAS_EARLYSUSPEND)
	struct early_suspend		early_suspend;
#endif
	struct touch_platform_data 	*pdata;
	struct touch_data		ts_data;
	struct touch_fw_info		fw_info;
	struct section_info		st_info;
	struct kobject 			lge_touch_kobj;
	struct ghost_finger_ctrl	gf_ctrl;
	bool sd_status;
	struct completion irq_completion;
};

#define ts_pdata	((ts)->pdata)
#define ts_caps		(ts_pdata->caps)
#define ts_role		(ts_pdata->role)
#define ts_pwr		(ts_pdata->pwr)
#define touch_enable(ts) enable_irq(ts->client->irq);
#define touch_disable(ts) disable_irq(ts->client->irq);

#define MAX_RETRY_COUNT		5
#define MAX_GHOST_CHECK_COUNT	3

struct touch_device_driver	*touch_drv;
static struct workqueue_struct	*touch_wq;

#define get_time_interval(a, b) 	(a >= b ? a - b : 1000000 + a - b)
#define jitter_abs(x)		(x >= 0 ? x : -x)
#define jitter_sub(x, y)	(x >= y ? x - y : y - x)

#ifdef LGE_TOUCH_GHOST_DETECTION
static unsigned int ta_debouncing_count;
static unsigned int button_press_count;
static unsigned int ts_rebase_count;
struct timeval t_ex_debug[TIME_EX_PROFILE_MAX];
struct t_data	 ts_prev_finger_press_data;
int long_press_check_count;
int force_continuous_mode;
int long_press_check;
int finger_subtraction_check_count;
bool ghost_detection;
int ghost_detection_count;
int trigger_baseline;
int ts_charger_plug;
#endif

int power_block;

static void safety_reset(struct lge_touch_data *ts);
static int touch_ic_init(struct lge_touch_data *ts);
static void touch_work_func_a(struct lge_touch_data *ts);

/* Debug mask value
 * usage: echo [debug_mask] > /sys/module/lge_ts_core/parameters/debug_mask
 */
u32 touch_debug_mask_ = 0
			| DEBUG_BASE_INFO
			/* | DEBUG_ABS */
			| DEBUG_CONFIG
			/* | DEBUG_TRACE */
			| DEBUG_FW_UPGRADE
			| DEBUG_ABS_POINT
			;
module_param_named(debug_mask, touch_debug_mask_, int, S_IRUGO|S_IWUSR|S_IWGRP);

static void release_all_ts_event(struct lge_touch_data *ts);

/* set_touch_handle_
 *
 * Developer can save their object using 'set_touch_handle_'.
 * Also, they can restore that using 'get_touch_handle_'.
 */

void set_touch_handle_(struct i2c_client *client, void *h_touch)
{
	struct lge_touch_data *ts = i2c_get_clientdata(client);
	ts->h_touch = h_touch;
}

void *get_touch_handle_(struct i2c_client *client)
{
	struct lge_touch_data *ts = i2c_get_clientdata(client);
	return ts->h_touch;
}

void power_lock(int value)
{
	power_block |= value;
}

void power_unlock(int value)
{
	power_block &= ~(value);
}

#ifdef LGE_TOUCH_GHOST_DETECTION
static enum hrtimer_restart touch_trigger_timer_handler(struct hrtimer *timer)
{
	struct lge_touch_data *ts =
			container_of(timer, struct lge_touch_data, trigger_timer);

	if (ts_role->ghost_detection_enable) {
		if (trigger_baseline == 1 && atomic_read(&ts->device_init) == 1) {
			trigger_baseline = 2;
			atomic_inc(&ts->next_work);
			queue_work(touch_wq, &ts->work);
		}
	}
	return HRTIMER_NORESTART;
}

void trigger_baseline_state_machine(int usb_type)
{
	u8 buf = 0;

	if (touch_test_dev && touch_test_dev->pdata->role->ghost_detection_enable) {
		if (touch_test_dev->curr_pwr_state == POWER_ON) {
			if (usb_type == 0) {
				touch_i2c_read(touch_test_dev->client, 0x50, 1, &buf);
				buf = buf & 0xDF;
				touch_i2c_write_byte(touch_test_dev->client, 0x50, buf);
			} else {
				touch_i2c_read(touch_test_dev->client, 0x50, 1, &buf);
				buf = buf | 0x20;
				touch_i2c_write_byte(touch_test_dev->client, 0x50, buf);
			}
		}
		TOUCH_INFO_MSG(" trigger_baseline_state_machine = %d \n", usb_type);

		ts_charger_plug = (usb_type == 0) ? 0 : 1;
		if (trigger_baseline == 0) {
			trigger_baseline = 1;
			hrtimer_start(&ts->trigger_timer, ktime_set(0, MS_TO_NS(1000)), HRTIMER_MODE_REL);
		}
	}
}


/* Ghost Detection Solution */
static u8 resume_flag;
int ghost_detect_solution(struct lge_touch_data *ts)
{
	int first_int_detection = 0;
	int cnt = 0, id = 0;

	if (ts->gf_ctrl.incoming_call && (ts->ts_data.total_num > 1)) {
		TOUCH_INFO_MSG("call state rebase\n");
		goto out_need_to_rebase;
	}

	if (trigger_baseline == 2)
		goto out_need_to_rebase;

	if (resume_flag) {
		resume_flag = 0;
		do_gettimeofday(&t_ex_debug[TIME_EX_FIRST_INT_TIME]);

		if (t_ex_debug[TIME_EX_FIRST_INT_TIME].tv_sec - t_ex_debug[TIME_EX_INIT_TIME].tv_sec == 0) {
			if ((get_time_interval(t_ex_debug[TIME_EX_FIRST_INT_TIME].tv_usec,
				t_ex_debug[TIME_EX_INIT_TIME].tv_usec)) <= 200000) {
					first_int_detection = 1;
				}
		} else if (t_ex_debug[TIME_EX_FIRST_INT_TIME].tv_sec - t_ex_debug[TIME_EX_INIT_TIME].tv_sec == 1) {
			if (t_ex_debug[TIME_EX_FIRST_INT_TIME].tv_usec + 1000000
				- t_ex_debug[TIME_EX_INIT_TIME].tv_usec <= 200000) {
				first_int_detection = 1;
			}
		}
	}

	if (first_int_detection) {
		for (cnt = 0; cnt < ts_caps->max_id; cnt++) {
			if (ts->ts_data.curr_data[cnt].status == FINGER_PRESSED) {
					TOUCH_INFO_MSG("ghost detected within first input time 200ms\n");
					ghost_detection = true;
			}
		}
	}

	if (ts_charger_plug) {
		if ((ts_role->ta_debouncing_finger_num  <= ts->ts_data.total_num) && (ta_debouncing_count < ts_role->ta_debouncing_count)) {
			ts_role->ta_debouncing_count++;
			memset(&ts->ts_data.curr_data, 0x0, sizeof(ts->ts_data.curr_data));
			goto out_need_to_debounce;
		} else if (ts->ts_data.total_num < ts_role->ta_debouncing_finger_num) {
			ts_role->ta_debouncing_count = 0;
		} else {
		}
	}

	if ((ts->ts_data.state != TOUCH_ABS_LOCK) && (ts->ts_data.total_num)) {

		if (ts->ts_data.prev_total_num != ts->ts_data.total_num) {
			if (ts->ts_data.prev_total_num <= ts->ts_data.total_num) {
				if (ts->gf_ctrl.stage == GHOST_STAGE_CLEAR || (ts->gf_ctrl.stage | GHOST_STAGE_1) || ts->gf_ctrl.stage == GHOST_STAGE_4)
					ts->ts_data.state = TOUCH_BUTTON_LOCK;

				for (id = 0; id < ts_caps->max_id; id++) {
					if (ts->ts_data.curr_data[id].status == FINGER_PRESSED
							&& ts->ts_data.prev_data[id].status == FINGER_RELEASED) {
						break;
					}
				}

				if (id < 10) {
					memcpy(&t_ex_debug[TIME_EX_PREV_PRESS_TIME], &t_ex_debug[TIME_EX_CURR_PRESS_TIME], sizeof(struct timeval));
					do_gettimeofday(&t_ex_debug[TIME_EX_CURR_PRESS_TIME]);

					if (1 <= ts->ts_data.prev_total_num && 1 <= ts->ts_data.total_num && jitter_sub(ts_prev_finger_press_data.x_position, ts->ts_data.curr_data[id].x_position) <= 10 && jitter_sub(ts_prev_finger_press_data.y_position, ts->ts_data.curr_data[id].y_position) <= 10) {
						/* if time_interval between prev fingger pressed and curr finger pressed is less than 50ms, we need to rebase touch ic. */
						if (((t_ex_debug[TIME_EX_CURR_PRESS_TIME].tv_sec - t_ex_debug[TIME_EX_PREV_PRESS_TIME].tv_sec) == 1) &&
							((get_time_interval(t_ex_debug[TIME_EX_CURR_PRESS_TIME].tv_usec + 1000000, t_ex_debug[TIME_EX_PREV_PRESS_TIME].tv_usec)) <= 50 * 1000)) {
							ghost_detection = true;
							ghost_detection_count++;
						} else if (((t_ex_debug[TIME_EX_CURR_PRESS_TIME].tv_sec - t_ex_debug[TIME_EX_PREV_PRESS_TIME].tv_sec) == 0) &&
							((get_time_interval(t_ex_debug[TIME_EX_CURR_PRESS_TIME].tv_usec, t_ex_debug[TIME_EX_PREV_PRESS_TIME].tv_usec)) <= 50 * 1000)) {
							ghost_detection = true;
							ghost_detection_count++;
						} else {
						}	/* do not anything */
					} else if (ts->ts_data.prev_total_num == 0 && ts->ts_data.total_num == 1 && jitter_sub(ts_prev_finger_press_data.x_position, ts->ts_data.curr_data[id].x_position) <= 10 && jitter_sub(ts_prev_finger_press_data.y_position, ts->ts_data.curr_data[id].y_position) <= 10) {
					       /* if time_interval between prev fingger pressed and curr finger pressed is less than 50ms, we need to rebase touch ic. */
						if (((t_ex_debug[TIME_EX_CURR_PRESS_TIME].tv_sec - t_ex_debug[TIME_EX_PREV_PRESS_TIME].tv_sec) == 1) &&
							((get_time_interval(t_ex_debug[TIME_EX_CURR_PRESS_TIME].tv_usec + 1000000, t_ex_debug[TIME_EX_PREV_PRESS_TIME].tv_usec)) <= 50 * 1000)) {
							ghost_detection = true;
						} else if (((t_ex_debug[TIME_EX_CURR_PRESS_TIME].tv_sec - t_ex_debug[TIME_EX_PREV_PRESS_TIME].tv_sec) == 0) &&
							((get_time_interval(t_ex_debug[TIME_EX_CURR_PRESS_TIME].tv_usec, t_ex_debug[TIME_EX_PREV_PRESS_TIME].tv_usec)) <= 50 * 1000)) {
							ghost_detection = true;
						} else {
						}	/* do not anything */
					} else if (5 < jitter_sub(ts->ts_data.prev_total_num, ts->ts_data.total_num)) {
						 ghost_detection = true;
					} else {
					}	/* do not anything */

					memcpy(&ts_prev_finger_press_data, &ts->ts_data.curr_data[id], sizeof(ts_prev_finger_press_data));
				}
			} else {
				memcpy(&t_ex_debug[TIME_EX_PREV_PRESS_TIME], &t_ex_debug[TIME_EX_CURR_PRESS_TIME], sizeof(struct timeval));
				do_gettimeofday(&t_ex_debug[TIME_EX_CURR_INT_TIME]);

				   /* if finger subtraction time is less than 10ms, we need to check ghost state. */
				if (((t_ex_debug[TIME_EX_CURR_INT_TIME].tv_sec - t_ex_debug[TIME_EX_PREV_PRESS_TIME].tv_sec) == 1) &&
					((get_time_interval(t_ex_debug[TIME_EX_CURR_INT_TIME].tv_usec + 1000000, t_ex_debug[TIME_EX_PREV_PRESS_TIME].tv_usec)) < 11 * 1000))
					finger_subtraction_check_count++;
				else if (((t_ex_debug[TIME_EX_CURR_INT_TIME].tv_sec - t_ex_debug[TIME_EX_PREV_PRESS_TIME].tv_sec) == 0) &&
					((get_time_interval(t_ex_debug[TIME_EX_CURR_INT_TIME].tv_usec, t_ex_debug[TIME_EX_PREV_PRESS_TIME].tv_usec)) < 11 * 1000))
					finger_subtraction_check_count++;
				else
					finger_subtraction_check_count = 0;

				if (4 < finger_subtraction_check_count) {
					finger_subtraction_check_count = 0;
					TOUCH_INFO_MSG("need_to_rebase finger_subtraction!!! \n");
					goto out_need_to_rebase;
				}
			}
		}
	} else if (!ts->ts_data.total_num) {
			long_press_check_count = 0;
			finger_subtraction_check_count = 0;
	}

	if (ts->ts_data.state != TOUCH_BUTTON_LOCK) {
		if (ts->work_sync_err_cnt > 0
				&& ts->ts_data.prev_button.state == BUTTON_RELEASED) {
			/* Do nothing */
		} else {
			if (button_press_count == 0)
				do_gettimeofday(&t_ex_debug[TIME_EX_BUTTON_PRESS_START_TIME]);
			else
				do_gettimeofday(&t_ex_debug[TIME_EX_BUTTON_PRESS_END_TIME]);

			button_press_count++;

			if (6 <= button_press_count) {
				 if (((t_ex_debug[TIME_EX_BUTTON_PRESS_END_TIME].tv_sec - t_ex_debug[TIME_EX_BUTTON_PRESS_START_TIME].tv_sec) == 1) &&
					((get_time_interval(t_ex_debug[TIME_EX_BUTTON_PRESS_END_TIME].tv_usec + 1000000, t_ex_debug[TIME_EX_BUTTON_PRESS_START_TIME].tv_usec)) <= 100 * 1000)) {
						TOUCH_INFO_MSG("need_to_rebase button zero\n");
						goto out_need_to_rebase;
				} else if (((t_ex_debug[TIME_EX_BUTTON_PRESS_END_TIME].tv_sec - t_ex_debug[TIME_EX_BUTTON_PRESS_START_TIME].tv_sec) == 0) &&
					((get_time_interval(t_ex_debug[TIME_EX_BUTTON_PRESS_END_TIME].tv_usec, t_ex_debug[TIME_EX_BUTTON_PRESS_START_TIME].tv_usec)) <= 100 * 1000)) {
						TOUCH_INFO_MSG("need_to_rebase button zero\n");
						goto out_need_to_rebase;
				} else {
				}	/* do not anything */

				button_press_count = 0;
			} else {
				if ((t_ex_debug[TIME_EX_BUTTON_PRESS_END_TIME].tv_sec -
					t_ex_debug[TIME_EX_BUTTON_PRESS_START_TIME].tv_sec) > 1)
						button_press_count = 0;
				else if (((t_ex_debug[TIME_EX_BUTTON_PRESS_END_TIME].tv_sec - t_ex_debug[TIME_EX_BUTTON_PRESS_START_TIME].tv_sec) == 1) &&
					((get_time_interval(t_ex_debug[TIME_EX_BUTTON_PRESS_END_TIME].tv_usec + 1000000, t_ex_debug[TIME_EX_BUTTON_PRESS_START_TIME].tv_usec)) >= 100 * 1000)) {
						button_press_count = 0;
				} else if (((t_ex_debug[TIME_EX_BUTTON_PRESS_END_TIME].tv_sec - t_ex_debug[TIME_EX_BUTTON_PRESS_START_TIME].tv_sec) == 0) &&
					((get_time_interval(t_ex_debug[TIME_EX_BUTTON_PRESS_END_TIME].tv_usec, t_ex_debug[TIME_EX_BUTTON_PRESS_START_TIME].tv_usec)) >= 100 * 1000)) {
						button_press_count = 0;
				} else {
				}	/*do not anything */
			}
		}
	}

	if (ghost_detection == true && ts->ts_data.total_num == 0 && ts->ts_data.palm == 0) {
		TOUCH_INFO_MSG("need_to_rebase zero\n");

		goto out_need_to_rebase;
	} else if (ghost_detection == true && 3 <= ghost_detection_count && ts->ts_data.palm == 0) {
		TOUCH_INFO_MSG("need_to_rebase zero 3\n");
		goto out_need_to_rebase;
	}

	return 0;

out_need_to_debounce:
	return NEED_TO_OUT;

out_need_to_rebase:
	{
		ghost_detection = false;
		ghost_detection_count = 0;
		memset(&ts_prev_finger_press_data, 0x0, sizeof(ts_prev_finger_press_data));
		button_press_count = 0;
		ts_rebase_count++;

		if (ts_rebase_count == 1) {
			do_gettimeofday(&t_ex_debug[TIME_EX_FIRST_GHOST_DETECT_TIME]);

			if ((t_ex_debug[TIME_EX_FIRST_GHOST_DETECT_TIME].tv_sec - t_ex_debug[TIME_EX_INIT_TIME].tv_sec) <= 3) {
				ts_rebase_count = 0;
				TOUCH_INFO_MSG("need_to_init in 3 sec\n");
				goto out_need_to_init;
			}
		} else {
			do_gettimeofday(&t_ex_debug[TIME_EX_SECOND_GHOST_DETECT_TIME]);

			if (((t_ex_debug[TIME_EX_SECOND_GHOST_DETECT_TIME].tv_sec - t_ex_debug[TIME_EX_FIRST_GHOST_DETECT_TIME].tv_sec) <= 5)) {
				ts_rebase_count = 0;
				TOUCH_INFO_MSG("need_to_init\n");
				goto out_need_to_init;
			} else {
				ts_rebase_count = 1;
				memcpy(&t_ex_debug[TIME_EX_FIRST_GHOST_DETECT_TIME], &t_ex_debug[TIME_EX_SECOND_GHOST_DETECT_TIME], sizeof(struct timeval));
			}
		}
		release_all_ts_event(ts);
		memset(&ts->ts_data, 0, sizeof(ts->ts_data));
		if (touch_drv->ic_ctrl) {
			if (touch_drv->ic_ctrl(ts->client, IC_CTRL_BASELINE, BASELINE_REBASE) < 0) {
				TOUCH_ERR_MSG("IC_CTRL_REBASE(2) handling fail\n");
			}
		}
		TOUCH_INFO_MSG("need_to_rebase\n");
	}
	return NEED_TO_OUT;

out_need_to_init:
	return NEED_TO_INIT;
}
#endif

#if 0
int touch_i2c_read(struct i2c_client *client, u8 reg, int len, u8 *buf)
{
	struct i2c_msg msgs[] = {
		{
			.addr = client->addr,
			.flags = 0,
			.len = 1,
			.buf = &reg,
		},
		{
			.addr = client->addr,
			.flags = I2C_M_RD,
			.len = len,
			.buf = buf,
		},
	};

	if (i2c_transfer(->adapter, msgs, 2) < 0) {
		if (printk_ratelimit())
			TOUCH_ERR_MSG("transfer error\n");
		return -EIO;
	} else
		return 0;
}

int touch_i2c_write(struct i2c_client *client, u8 reg, int len, u8  *buf)
{
	unsigned char send_buf[len + 1];
		struct i2c_msg msgs[] = {
			{
				.addr = client->addr,
				.flags = client->flags,
				.len = len+1,
			.buf = send_buf,
		},
	};

	send_buf[0] = (unsigned char)reg;
	memcpy(&send_buf[1], buf, len);

	if (i2c_transfer(->adapter, msgs, 1) < 0) {
		if (printk_ratelimit())
			TOUCH_ERR_MSG("transfer error\n");
		return -EIO;
	} else
		return 0;
}
#endif


/* touch_asb_input_report
 *
 * finger status report
 */
static int touch_asb_input_report(struct lge_touch_data *ts, int status)
{
	int id = 0;
	u8 total = 0;

	struct input_dev *input_dev = ts->input_dev;

	if (status == FINGER_PRESSED) {
		for (id = 0; id < ts_caps->max_id; id++) {
			if ((ts_role->key_type == TOUCH_SOFT_KEY)
					&& (ts->ts_data.curr_data[id].y_position >= ts_caps->y_button_boundary))
				continue;

			if (ts->ts_data.curr_data[id].status == FINGER_PRESSED) {
				input_mt_slot(input_dev, id);
				input_mt_report_slot_state(input_dev, MT_TOOL_FINGER, true);
				input_report_abs(input_dev, ABS_MT_POSITION_X,
						 ts->ts_data.curr_data[id].x_position);

				/* When a user's finger cross the boundary (from key to LCD),
					a ABS-event will change its y-position to edge of LCD, automatically.*/

				if (ts_role->key_type == TOUCH_SOFT_KEY
						&& ts->ts_data.curr_data[id].y_position < ts_caps->y_button_boundary
						&& ts->ts_data.prev_data[id].y_position > ts_caps->y_button_boundary
						&& ts->ts_data.prev_button.key_code != KEY_NULL)
					input_report_abs(input_dev, ABS_MT_POSITION_Y, ts_caps->y_button_boundary);
				else
					input_report_abs(input_dev, ABS_MT_POSITION_Y, ts->ts_data.curr_data[id].y_position);

				if (ts_caps->is_pressure_supported)
					input_report_abs(input_dev, ABS_MT_PRESSURE,
							ts->ts_data.curr_data[id].pressure);

				if (ts_caps->is_width_supported) {
					input_report_abs(input_dev, ABS_MT_WIDTH_MAJOR,
							ts->ts_data.curr_data[id].width_major);
					input_report_abs(input_dev, ABS_MT_WIDTH_MINOR,
							ts->ts_data.curr_data[id].width_minor);
					input_report_abs(input_dev, ABS_MT_ORIENTATION,
							ts->ts_data.curr_data[id].width_orientation);
				}

				if (ts_caps->is_id_supported)
					input_report_abs(input_dev, ABS_MT_TRACKING_ID, id);

				ts->ts_data.curr_data[id].status = FINGER_HOLD;
				total++;

				if (unlikely(touch_debug_mask_ & DEBUG_ABS))
					TOUCH_INFO_MSG("hyj-press[%d] pos[%4d,%4d] w_m[%2d] w_n[%2d] w_o[%2d] p[%3d]\n",
							ts_caps->is_id_supported ? id : 0,
							ts->ts_data.curr_data[id].x_position,
							ts->ts_data.curr_data[id].y_position,
							ts_caps->is_width_supported ? ts->ts_data.curr_data[id].width_major : 0,
							ts_caps->is_width_supported ? ts->ts_data.curr_data[id].width_minor : 0,
							ts_caps->is_width_supported ? ts->ts_data.curr_data[id].width_orientation : 0,
							ts_caps->is_pressure_supported ? ts->ts_data.curr_data[id].pressure : 0);
			} else {
				/* release handling */
				if (ts_role->key_type != TOUCH_SOFT_KEY && ts->ts_data.curr_data[id].status == FINGER_RELEASED) {
					if (unlikely(touch_debug_mask_ & DEBUG_ABS))
						TOUCH_INFO_MSG("hyj-release [%d]\n", id);
					ts->ts_data.curr_data[id].status = FINGER_UNUSED;
					memset(&ts->ts_data.prev_data[id], 0x0, sizeof(ts->ts_data.prev_data[id]));

					input_mt_slot(input_dev, id);
					input_mt_report_slot_state(input_dev, MT_TOOL_FINGER, false);
				}
			}
		}
	} else if (status == FINGER_RELEASED) {
		for (id = 0; id < ts_caps->max_id; id++) {
			if (ts->ts_data.curr_data[id].status == FINGER_RELEASED) {
					if (unlikely(touch_debug_mask_ & DEBUG_ABS))
						TOUCH_INFO_MSG("hyj-release [%d]R\n", id);
				ts->ts_data.curr_data[id].status = FINGER_UNUSED;

				memset(&ts->ts_data.prev_data[id], 0x0, sizeof(ts->ts_data.prev_data[id]));

				input_mt_slot(input_dev, id);
				input_mt_report_slot_state(input_dev, MT_TOOL_FINGER, false);
			}
		}
	} else if (status == ALL_FINGER_RELEASED) {
		for (id = 0; id < ts_caps->max_id; id++) {
			if (ts->ts_data.curr_data[id].status >= FINGER_PRESSED) {
				TOUCH_INFO_MSG("touch_release[%s] : <%d> x[%4d] y[%4d]\n",
					ts->ts_data.palm ? "Palm" : " ", id,
					ts->ts_data.curr_data[id].x_position,
					ts->ts_data.curr_data[id].y_position);
			}

			ts->ts_data.curr_data[id].status = FINGER_UNUSED;
			ts->ts_data.curr_data[id].point_log_state = 0;
			memset(&ts->ts_data.prev_data[id], 0x0, sizeof(ts->ts_data.prev_data[id]));

			input_mt_slot(input_dev, id);
			input_mt_report_slot_state(input_dev, MT_TOOL_FINGER, false);
		}
	}

	return total;
}

static char *get_touch_button_string(u16 key_code)
{
	static char str[16] = {0};

	switch (key_code) {
	case KEY_BACK: /*158 0x9E*/
		sprintf(str, "BACK");
		break;
	case KEY_HOMEPAGE: /*172 0xAC*/
		sprintf(str, "HOME");
		break;
	case KEY_MENU: /* 139 0x8B*/
		sprintf(str, "MENU");
		break;
	case KEY_SIMSWITCH: /*249 0xF9*/
		sprintf(str, "SIM_SWITCH");
		break;
	default:
		sprintf(str, "Unknown");
		break;
	}

	return str;
}

/* release_all_ts_event
 *
 * When system enters suspend-state,
 * if user press touch-panel, release them automatically.
 */
static void release_all_ts_event(struct lge_touch_data *ts)
{
	if (ts->input_dev == NULL) {
		TOUCH_DEBUG_MSG("Input device is NOT allocated!!\n");
		return;
	}

	if (ts_role->key_type == TOUCH_HARD_KEY) {
		touch_asb_input_report(ts, ALL_FINGER_RELEASED);

		if (likely(touch_debug_mask_ & (DEBUG_ABS)))
			TOUCH_INFO_MSG("touch finger position released\n");

		if (ts->ts_data.prev_button.state == BUTTON_PRESSED) {
			input_report_key(ts->input_dev, ts->ts_data.prev_button.key_code, BUTTON_RELEASED);

			if (likely(touch_debug_mask_ & (DEBUG_BUTTON | DEBUG_BASE_INFO)))
				TOUCH_INFO_MSG("KEY[%s:%3d] is released\n",
					get_touch_button_string(ts->ts_data.prev_button.key_code), ts->ts_data.prev_button.key_code);
		}
	} else if (ts_role->key_type == VIRTUAL_KEY) {
		if (ts->ts_data.prev_total_num) {
			touch_asb_input_report(ts, FINGER_RELEASED);

			if (likely(touch_debug_mask_ & (DEBUG_ABS | DEBUG_BASE_INFO)))
				TOUCH_INFO_MSG("touch finger position released\n");
		}
	} else if (ts_role->key_type == TOUCH_SOFT_KEY) {
		if (ts->ts_data.state == ABS_PRESS) {
			touch_asb_input_report(ts, FINGER_RELEASED);

			if (likely(touch_debug_mask_ & (DEBUG_ABS | DEBUG_BASE_INFO)))
				TOUCH_INFO_MSG("touch finger position released\n");
		} else if (ts->ts_data.state == BUTTON_PRESS) {
			input_report_key(ts->input_dev, ts->ts_data.prev_button.key_code, BUTTON_RELEASED);

			if (likely(touch_debug_mask_ & (DEBUG_BUTTON | DEBUG_BASE_INFO)))
				TOUCH_INFO_MSG("KEY[%d] is released\n", ts->ts_data.prev_button.key_code);
		}
	}

	ts->ts_data.prev_total_num = 0;
	ts->ts_data.touch_count_num = 0;
	input_sync(ts->input_dev);
}

/* touch_power_cntl
 *
 * 1. POWER_ON
 * 2. POWER_OFF
 * 3. POWER_SLEEP
 * 4. POWER_WAKE
 */
static int touch_power_cntl(struct lge_touch_data *ts, int onoff)
{
	int ret = 0;

	if (touch_drv->power == NULL) {
		TOUCH_INFO_MSG("There is no specific power control function\n");
		return -1;
	}

	switch (onoff) {
	case POWER_ON:
		ret = touch_drv->power(ts->client, POWER_ON);
		if (ret < 0) {
			TOUCH_ERR_MSG("power on failed\n");
		} else {
			ts->curr_pwr_state = POWER_ON;
			TOUCH_INFO_MSG("power on \n");
		}
		break;

	case POWER_OFF:
		ret = touch_drv->power(ts->client, POWER_OFF);
		if (ret < 0) {
			TOUCH_ERR_MSG("power off failed\n");
		} else {
			ts->curr_pwr_state = POWER_OFF;
			TOUCH_INFO_MSG("power off \n");
		}

		msleep(ts_role->reset_delay);
		atomic_set(&ts->device_init, 0);
		break;

	case POWER_SLEEP:
		ret = touch_drv->power(ts->client, POWER_SLEEP);
		if (ret < 0) {
			TOUCH_ERR_MSG("power sleep failed\n");
		} else {
			ts->curr_pwr_state = POWER_SLEEP;
			TOUCH_INFO_MSG("power sleep \n");
		}
		break;

	case POWER_WAKE:
		ret = touch_drv->power(ts->client, POWER_WAKE);
		if (ret < 0) {
			TOUCH_ERR_MSG("power wake failed\n");
		} else {
			ts->curr_pwr_state = POWER_WAKE;
			TOUCH_INFO_MSG("power wake \n");
		}
		break;

	default:
		break;
	}

	if (ret >= 0)
		TOUCH_POWER_MSG("power_state[%d]", ts->curr_pwr_state);

	return ret;
}


/* safety_reset
 *
 * 1. disable irq/timer.
 * 2. turn off the power.
 * 3. turn on the power.
 * 4. sleep (booting_delay)ms, usually 400ms(synaptics).
 * 5. enable irq/timer.
 *
 * After 'safety_reset', we should call 'touch_init'.
 */
static void safety_reset(struct lge_touch_data *ts)
{
	touch_disable(ts);

#ifdef LGE_TOUCH_GHOST_DETECTION
	if (ts_role->ghost_detection_enable) {
		hrtimer_cancel(&ts->trigger_timer);
	}
#endif

	release_all_ts_event(ts);

	touch_power_cntl(ts, POWER_OFF);
	touch_power_cntl(ts, POWER_ON);
	msleep(ts_role->booting_delay);
	touch_enable(ts);

	return;
}

/* touch_ic_init
 *
 * initialize the device_IC and variables.
 */
static int touch_ic_init(struct lge_touch_data *ts)
{
	/*
	 * TODO
	 * - initialize ghost, jitter, accuracy
	 */

	int pinstate[3] = {0}; /* SDA, SCL, INT */

	if (unlikely(ts->ic_init_err_cnt >= MAX_RETRY_COUNT)) {
		TOUCH_ERR_MSG("Init Failed: Irq-pin has some unknown problems\n");
		goto err_out_critical;
	}

	atomic_set(&ts->device_init, 1);

	if (touch_drv->init == NULL) {
		TOUCH_INFO_MSG("There is no specific IC init function\n");
		goto err_out_critical;
	}

	if (gpio_is_valid(ts_pdata->sda_pin))
		pinstate[0] = gpio_get_value(ts_pdata->sda_pin);

	if (gpio_is_valid(ts_pdata->scl_pin))
		pinstate[1] = gpio_get_value(ts_pdata->scl_pin);

	if (gpio_is_valid(ts_pdata->int_pin))
		pinstate[2] = gpio_get_value(ts_pdata->int_pin);

	if (pinstate[0] == 0 || pinstate[1] == 0 || pinstate[2] == 0)
	TOUCH_INFO_MSG("pin state [sda:%d, scl:%d, int:%d]\n", pinstate[0], pinstate[1], pinstate[2]);

#if 0
	if (touch_drv->init(ts->client, &ts->fw_info) < 0) {
		TOUCH_ERR_MSG("specific device initialization fail\n");
		goto err_out_retry;
	}
#endif

#ifdef LGE_TOUCH_GHOST_DETECTION
	if (ts_role->ghost_detection_enable) {
		/* force continuous mode after IC init  */
		if (touch_drv->ic_ctrl) {
			TOUCH_INFO_MSG("force continuous mode !!!\n");
			if (touch_drv->ic_ctrl(ts->client, IC_CTRL_REPORT_MODE, 0) < 0) {
				TOUCH_ERR_MSG("IC_CTRL_BASELINE handling fail\n");
				goto err_out_retry;
			}
			force_continuous_mode = 1;
		}
		trigger_baseline = 0;
		ghost_detection = 0;
		ghost_detection_count = 0;
		do_gettimeofday(&t_ex_debug[TIME_EX_INIT_TIME]);

		ts->gf_ctrl.count = 0;
		ts->gf_ctrl.ghost_check_count = 0;
		ts->gf_ctrl.saved_x = -1;
		ts->gf_ctrl.saved_y = -1;

		if (ts->gf_ctrl.probe) {
			ts->gf_ctrl.stage = GHOST_STAGE_1;
			if (touch_drv->ic_ctrl) {
				if (touch_drv->ic_ctrl(ts->client, IC_CTRL_BASELINE, BASELINE_OPEN) < 0) {
					TOUCH_ERR_MSG("IC_CTRL_BASELINE handling fail\n");
					goto err_out_retry;
				}
			}
		} else {
			if (ts->gf_ctrl.stage & GHOST_STAGE_2) {
				ts->gf_ctrl.stage = GHOST_STAGE_1 | GHOST_STAGE_2 | GHOST_STAGE_4;
				ts->gf_ctrl.ghost_check_count = MAX_GHOST_CHECK_COUNT - 1;
				if (touch_drv->ic_ctrl) {
					if (touch_drv->ic_ctrl(ts->client, IC_CTRL_BASELINE, BASELINE_OPEN) < 0) {
						TOUCH_ERR_MSG("IC_CTRL_BASELINE handling fail\n");
						goto err_out_retry;
					}
				}
			} else {
				ts->gf_ctrl.stage = GHOST_STAGE_3;
				if (touch_drv->ic_ctrl) {
					if (touch_drv->ic_ctrl(ts->client, IC_CTRL_BASELINE, BASELINE_FIX) < 0) {
						TOUCH_ERR_MSG("IC_CTRL_BASELINE handling fail\n");
						goto err_out_retry;
					}
				}
			}
		}
	}
#endif

	if (unlikely(touch_debug_mask_ & (DEBUG_BASE_INFO))) {
#if 0
		TOUCH_INFO_MSG("%s %s\n",
				ts_pdata->maker, (ts_pdata->ic_type ? MMS_100A : MMS_100S));
#endif
		TOUCH_INFO_MSG("ghost_stage[0x%x]\n", ts->gf_ctrl.stage);
#if 0
		TOUCH_INFO_MSG("fw_image[%s]\n", ts_pdata->fw_image);
#endif
	}

	ts->gf_ctrl.probe = 0;

	memset(&ts->ts_data, 0, sizeof(ts->ts_data));
	ts->fw_info.is_downloading = 0;
	ts->ic_init_err_cnt = 0;

	return 0;

#ifdef LGE_TOUCH_GHOST_DETECTION
err_out_retry:
	safety_reset(ts);
	ts->ic_init_err_cnt++;
	queue_delayed_work(touch_wq, &ts->work_init, msecs_to_jiffies(20));
	return 0;
#endif

err_out_critical:
	ts->ic_init_err_cnt = 0;

	return -1;
}

/* ghost_finger_solution
 *
 * GHOST_STAGE_1
 * - melt_mode.
 * - If user press and release their finger in 1 sec, STAGE_1 will be cleared. --> STAGE_2
 * - If there is no key-guard, ghost_finger_solution is finished.
 *
 * GHOST_STAGE_2
 * - no_melt_mode
 * - if user use multi-finger, stage will be changed to STAGE_1
 *   (We assume that ghost-finger occured)
 * - if key-guard is unlocked, STAGE_2 is cleared. --> STAGE_3
 *
 * GHOST_STAGE_3
 * - when user release their finger, device re-scan the baseline.
 * - Then, GHOST_STAGE3 is cleared and ghost_finger_solution is finished.
 */
#define ghost_sub(x, y)	(x > y ? x - y : y - x)

int ghost_finger_solution(struct lge_touch_data *ts)
{
	u8	id = 0;

	for (id = 0; id < ts_caps->max_id; id++) {
		if (ts->ts_data.curr_data[id].status == FINGER_PRESSED) {
			break;
		}
	}

	if (ts->gf_ctrl.stage & GHOST_STAGE_1) {
		if (ts->ts_data.total_num == 0 && ts->ts_data.curr_button.state == 0 && ts->ts_data.palm == 0) {
			if (ts->gf_ctrl.count < ts->gf_ctrl.min_count || ts->gf_ctrl.count >= ts->gf_ctrl.max_count) {
				if (ts->gf_ctrl.stage & GHOST_STAGE_2)
					ts->gf_ctrl.ghost_check_count = MAX_GHOST_CHECK_COUNT - 1;
				else
					ts->gf_ctrl.ghost_check_count = 0;
			} else {
				if (ghost_sub(ts->gf_ctrl.saved_x, ts->gf_ctrl.saved_last_x) > ts->gf_ctrl.max_moved ||
				   ghost_sub(ts->gf_ctrl.saved_y, ts->gf_ctrl.saved_last_y) > ts->gf_ctrl.max_moved)
					ts->gf_ctrl.ghost_check_count = MAX_GHOST_CHECK_COUNT;
				else
					ts->gf_ctrl.ghost_check_count++;

				if (unlikely(touch_debug_mask_ & DEBUG_GHOST))
					TOUCH_INFO_MSG("ghost_stage_1: delta[%d/%d/%d]\n",
						ghost_sub(ts->gf_ctrl.saved_x, ts->gf_ctrl.saved_last_x),
						ghost_sub(ts->gf_ctrl.saved_y, ts->gf_ctrl.saved_last_y),
						ts->gf_ctrl.max_moved);
			}

			if (unlikely(touch_debug_mask_ & DEBUG_GHOST || touch_debug_mask_ & DEBUG_BASE_INFO))
				TOUCH_INFO_MSG("ghost_stage_1: ghost_check_count+[0x%x]\n", ts->gf_ctrl.ghost_check_count);

			if (ts->gf_ctrl.ghost_check_count >= MAX_GHOST_CHECK_COUNT) {
				ts->gf_ctrl.ghost_check_count = 0;
				if (touch_drv->ic_ctrl) {
					if (touch_drv->ic_ctrl(ts->client, IC_CTRL_BASELINE, BASELINE_FIX) < 0)
						return -1;
				}
				ts->gf_ctrl.stage &= ~GHOST_STAGE_1;
				if (unlikely(touch_debug_mask_ & DEBUG_GHOST || touch_debug_mask_ & DEBUG_BASE_INFO))
					TOUCH_INFO_MSG("ghost_stage_1: cleared[0x%x]\n", ts->gf_ctrl.stage);
				if (!ts->gf_ctrl.stage) {
					if (unlikely(touch_debug_mask_ & DEBUG_GHOST))
						TOUCH_INFO_MSG("ghost_stage_finished. (NON-KEYGUARD)\n");
				}
			}
			ts->gf_ctrl.count = 0;
			ts->gf_ctrl.saved_x = -1;
			ts->gf_ctrl.saved_y = -1;
		} else if (ts->ts_data.total_num == 1 && ts->ts_data.curr_button.state == 0
				&& id == 0 && ts->ts_data.palm == 0) {
			if (ts->gf_ctrl.saved_x == -1 && ts->gf_ctrl.saved_x == -1) {
				ts->gf_ctrl.saved_x = ts->ts_data.curr_data[id].x_position;
				ts->gf_ctrl.saved_y = ts->ts_data.curr_data[id].y_position;
			}
			ts->gf_ctrl.count++;
			if (unlikely(touch_debug_mask_ & DEBUG_GHOST))
				TOUCH_INFO_MSG("ghost_stage_1: int_count[%d/%d]\n", ts->gf_ctrl.count, ts->gf_ctrl.max_count);
		} else {
			if (unlikely(touch_debug_mask_ & DEBUG_GHOST) && ts->gf_ctrl.count != ts->gf_ctrl.max_count)
				TOUCH_INFO_MSG("ghost_stage_1: Not good condition. total[%d] button[%d] id[%d] palm[%d]\n",
						ts->ts_data.total_num, ts->ts_data.curr_button.state,
						id, ts->ts_data.palm);
			ts->gf_ctrl.count = ts->gf_ctrl.max_count;
		}
	} else if (ts->gf_ctrl.stage & GHOST_STAGE_2) {
		if (ts->ts_data.total_num > 1 || (ts->ts_data.total_num == 1 && ts->ts_data.curr_button.state)) {
			ts->gf_ctrl.stage |= GHOST_STAGE_1;
			ts->gf_ctrl.ghost_check_count = MAX_GHOST_CHECK_COUNT - 1;
			ts->gf_ctrl.count = 0;
			ts->gf_ctrl.saved_x = -1;
			ts->gf_ctrl.saved_y = -1;
			if (touch_drv->ic_ctrl) {
				if (touch_drv->ic_ctrl(ts->client, IC_CTRL_BASELINE, BASELINE_OPEN) < 0)
					return -1;
#ifdef CUST_G_TOUCH
/* do nothing */
#else
				if (touch_drv->ic_ctrl(ts->client, IC_CTRL_BASELINE, BASELINE_REBASE) < 0)
					return -1;
#endif
			}
			if (unlikely(touch_debug_mask_ & DEBUG_GHOST))
				TOUCH_INFO_MSG("ghost_stage_2: multi_finger. return to ghost_stage_1[0x%x]\n", ts->gf_ctrl.stage);
		}
	} else if (ts->gf_ctrl.stage & GHOST_STAGE_3) {
		if (ts->ts_data.total_num == 0 && ts->ts_data.curr_button.state == 0 && ts->ts_data.palm == 0) {
			ts->gf_ctrl.ghost_check_count++;

			if (unlikely(touch_debug_mask_ & DEBUG_GHOST || touch_debug_mask_ & DEBUG_BASE_INFO))
				TOUCH_INFO_MSG("ghost_stage_3: ghost_check_count+[0x%x]\n", ts->gf_ctrl.ghost_check_count);

			if (ts->gf_ctrl.ghost_check_count >= MAX_GHOST_CHECK_COUNT) {
				ts->gf_ctrl.stage &= ~GHOST_STAGE_3;
				if (unlikely(touch_debug_mask_ & DEBUG_GHOST || touch_debug_mask_ & DEBUG_BASE_INFO))
					TOUCH_INFO_MSG("ghost_stage_3: cleared[0x%x]\n", ts->gf_ctrl.stage);
				if (!ts->gf_ctrl.stage) {
					if (unlikely(touch_debug_mask_ & DEBUG_GHOST))
						TOUCH_INFO_MSG("ghost_stage_finished. (NON-KEYGUARD)\n");
				}
			}
		} else if (ts->ts_data.total_num == 1 && ts->ts_data.curr_button.state == 0
				&& id == 0 && ts->ts_data.palm == 0)
				;
		else {
			ts->gf_ctrl.stage &= ~GHOST_STAGE_3;
			ts->gf_ctrl.stage |= GHOST_STAGE_1;
			ts->gf_ctrl.ghost_check_count = 0;
			ts->gf_ctrl.count = 0;
			ts->gf_ctrl.saved_x = -1;
			ts->gf_ctrl.saved_y = -1;
			if (touch_drv->ic_ctrl) {
				if (touch_drv->ic_ctrl(ts->client, IC_CTRL_BASELINE, BASELINE_OPEN) < 0)
					return -1;
			}

			if (unlikely(touch_debug_mask_ & DEBUG_GHOST)) {
				TOUCH_INFO_MSG("ghost_stage_3: Not good condition. total[%d] button[%d] id[%d] palm[%d]\n",
						ts->ts_data.total_num, ts->ts_data.curr_button.state,
						id, ts->ts_data.palm);
				TOUCH_INFO_MSG("ghost_stage_3: return to ghost_stage_1[0x%x]\n", ts->gf_ctrl.stage);
			}
		}
	} else if (ts->gf_ctrl.stage & GHOST_STAGE_4) {
		if (ts->ts_data.total_num == 0 && ts->ts_data.curr_button.state == 0 && ts->ts_data.palm == 0) {
#ifdef CUST_G_TOUCH
/* do nothing */
#else
			if (touch_drv->ic_ctrl) {
				if (touch_drv->ic_ctrl(ts->client, IC_CTRL_BASELINE, BASELINE_REBASE) < 0)
					return -1;
			}
#endif
			ts->gf_ctrl.stage = GHOST_STAGE_CLEAR;
			if (unlikely(touch_debug_mask_ & DEBUG_GHOST || touch_debug_mask_ & DEBUG_BASE_INFO))
				TOUCH_INFO_MSG("ghost_stage_4: cleared[0x%x]\n", ts->gf_ctrl.stage);
			if (unlikely(touch_debug_mask_ & DEBUG_GHOST))
				TOUCH_INFO_MSG("ghost_stage_finished. (KEYGUARD)\n");
		}

		if (unlikely(touch_debug_mask_ & DEBUG_GHOST) && ts->ts_data.palm != 0)
			TOUCH_INFO_MSG("ghost_stage_4: palm[%d]\n", ts->ts_data.palm);
	}

	ts->gf_ctrl.saved_last_x = ts->ts_data.curr_data[id].x_position;
	ts->gf_ctrl.saved_last_y = ts->ts_data.curr_data[id].y_position;

	return 0;
}


/* touch_init_func
 *
 * In order to reduce the booting-time,
 * we used delayed_work_queue instead of msleep or mdelay.
 */
static void touch_init_func(struct work_struct *work_init)
{
	struct lge_touch_data *ts =
			container_of(to_delayed_work(work_init), struct lge_touch_data, work_init);

	if (unlikely(touch_debug_mask_ & DEBUG_TRACE))
		TOUCH_DEBUG_MSG("\n");

	/* Specific device initialization */
	touch_ic_init(ts);
}

static void touch_lock_func(struct work_struct *work_touch_lock)
{
	struct lge_touch_data *ts =
			container_of(to_delayed_work(work_touch_lock), struct lge_touch_data, work_touch_lock);

	if (unlikely(touch_debug_mask_ & DEBUG_TRACE))
		TOUCH_DEBUG_MSG("\n");

	ts->ts_data.state = DO_NOT_ANYTHING;
}

/* check_log_finger_changed
 *
 * Check finger state change for Debug
 */
static void check_log_finger_changed(struct lge_touch_data *ts, u8 total_num)
{
	u16 tmp_p = 0;
	u16 tmp_r = 0;
	u16 id = 0;

#if 0
	TOUCH_INFO_MSG("%s %d ts->ts_data.prev_total_num=%d total_num=%d", __func__, __LINE__, ts->ts_data.prev_total_num, total_num);


	if (ts->ts_data.prev_total_num != total_num) {
		/* Finger added */
		if (ts->ts_data.prev_total_num <= total_num) {
			for (id = 0; id < ts_caps->max_id; id++) {
				if (ts->ts_data.curr_data[id].status == FINGER_PRESSED
						&& ts->ts_data.prev_data[id].status == FINGER_RELEASED) {
					break;
				}
			}
			TOUCH_INFO_MSG("%d finger pressed : <%d> x[%4d] y[%4d] z[%3d]\n",
					total_num, id,
					ts->ts_data.curr_data[id].x_position,
					ts->ts_data.curr_data[id].y_position,
					ts->ts_data.curr_data[id].pressure);
		} else {
		/* Finger subtracted */
			TOUCH_INFO_MSG("%d finger pressed\n", total_num);
		}
	}
#endif

	if (ts->ts_data.prev_total_num == total_num && total_num == 1) {
		/* Finger changed at one finger status - IC bug check */
		for (id = 0, tmp_p = 0; id < ts_caps->max_id; id++) {
			/* find pressed */
			if (ts->ts_data.curr_data[id].status == FINGER_PRESSED
					&& ts->ts_data.prev_data[id].status == FINGER_RELEASED) {
				tmp_p = id;
			}
			/* find released */
			if (ts->ts_data.curr_data[id].status == FINGER_RELEASED
					&& ts->ts_data.prev_data[id].status == FINGER_PRESSED) {
				tmp_r = id;
			}
		}

		if (tmp_p != tmp_r
				&& (ts->ts_data.curr_data[tmp_p].status
						!= ts->ts_data.prev_data[tmp_p].status)) {
			TOUCH_INFO_MSG("%d finger changed : <%d -> %d> x[%4d] y[%4d] z[%3d]\n",
						total_num, tmp_r, tmp_p,
						ts->ts_data.curr_data[id].x_position,
						ts->ts_data.curr_data[id].y_position,
						ts->ts_data.curr_data[id].pressure);
		}
	}
}

/* check_log_finger_released
 *
 * Check finger state change for Debug
 */
#if 0
static void check_log_finger_released(struct lge_touch_data *ts)
{
	u16 id = 0;

	for (id = 0; id < ts_caps->max_id; id++) {
		if (ts->ts_data.prev_data[id].status == FINGER_PRESSED) {
			break;
		}
	}

	TOUCH_INFO_MSG("touch_release[%s] : <%d> x[%4d] y[%4d]\n",
			ts->ts_data.palm ? "Palm" : "", id,
			ts->ts_data.prev_data[id].x_position,
			ts->ts_data.prev_data[id].y_position);
}
#endif

/* touch_work_pre_proc
 *
 * Pre-process work at touch_work
 */
static int touch_work_pre_proc(struct lge_touch_data *ts)
{
	int result = 0;

	ts->ts_data.total_num = 0;

	if (unlikely(ts->work_sync_err_cnt >= MAX_RETRY_COUNT)) {
		TOUCH_ERR_MSG("Work Sync Failed: Irq-pin has some unknown problems\n");
		return -EIO;
	}

#ifdef LGE_TOUCH_TIME_DEBUG
	do_gettimeofday(&t_debug[TIME_WORKQUEUE_START]);
#endif
	TOUCH_TRACE_FUNC();

	result = touch_drv->data(ts->client, &ts->ts_data);

	if (result == -EIO) {
		TOUCH_ERR_MSG("get data fail\n");
		return -EIO;
	} else if (result == -ENXIO) {
		if (++ts->ic_error_cnt > MAX_RETRY_COUNT) {
			TOUCH_INFO_MSG("Error cannot recover \n");
			disable_irq_nosync(ts->client->irq);
			release_all_ts_event(ts);
			TOUCH_INFO_MSG("disable_irq . \n");
			touch_power_cntl(ts, POWER_OFF);
			return -ENXIO;
		}
		TOUCH_INFO_MSG("Error or ESD detected. Retry(%d/%d) \n", ts->ic_error_cnt, MAX_RETRY_COUNT);
		return -EIO;
	}

	/* Ghost finger solution */
	if (likely(ts_role->ghost_finger_solution_enable) && unlikely(ts->gf_ctrl.stage)) {
		if (ghost_finger_solution(ts)) {
			TOUCH_ERR_MSG("ghost_finger_solution was failed\n");
			return -EIO;
		}
	}

	return 0;
}

/* touch_work_post_proc
 *
 * Post-process work at touch_work
 */
static void touch_work_post_proc(struct lge_touch_data *ts, int post_proc)
{

	if (post_proc >= WORK_POST_MAX)
		return;

	switch (post_proc) {
	case WORK_POST_OUT:
#ifdef LGE_TOUCH_TIME_DEBUG
		do_gettimeofday(&t_debug[TIME_WORKQUEUE_END]);
		time_profile_result(ts);
#endif

		ts->work_sync_err_cnt = 0;
		post_proc = WORK_POST_COMPLATE;
		break;

	case WORK_POST_ERR_RETRY:
		ts->work_sync_err_cnt++;
		/* safety_reset(ts); */
		release_all_ts_event(ts);
		touch_power_cntl(ts, POWER_OFF);
		touch_power_cntl(ts, POWER_ON);
		msleep(ts_role->booting_delay);
		touch_ic_init(ts);
		post_proc = WORK_POST_COMPLATE;
		break;

	case WORK_POST_ERR_CIRTICAL:
		ts->work_sync_err_cnt = 0;
		/* safety_reset(ts); */
		release_all_ts_event(ts);
		touch_power_cntl(ts, POWER_OFF);
		touch_power_cntl(ts, POWER_ON);
		msleep(ts_role->booting_delay);
		touch_ic_init(ts);
		post_proc = WORK_POST_COMPLATE;
		break;

	default:
		post_proc = WORK_POST_COMPLATE;
		break;
	}

	if (post_proc != WORK_POST_COMPLATE)
		touch_work_post_proc(ts, post_proc);
}

/* touch_work_func_a
 *
 * HARD_TOUCH_KEY
 */
static void touch_work_func_a(struct lge_touch_data *ts)
{
	u8 report_enable = 0;
	int ret = 0;

	ret = touch_work_pre_proc(ts);
	if (ret == -EIO)
		goto err_out_critical;
	else if (ret == -EAGAIN)
		goto out;
	else if (ret == -ENXIO)
		goto power_off;

#ifdef LGE_TOUCH_GHOST_DETECTION
	/* Ghost detection solution */
	if (ts_role->ghost_detection_enable) {
		ret = ghost_detect_solution(ts);
		if (ret == NEED_TO_OUT)
			goto out;
		else if (ret == NEED_TO_INIT)
			goto err_out_init;
	}
#endif

	/* Palm handle */
	if (ts->ts_data.palm) {
		release_all_ts_event(ts);
		cancel_delayed_work_sync(&ts->work_touch_lock);
		goto out;
	}

	/* Finger handle */
	if (!ts->ts_data.total_num) {
		touch_asb_input_report(ts, FINGER_RELEASED);
		report_enable = 1;

		queue_delayed_work(touch_wq, &ts->work_touch_lock, msecs_to_jiffies(200));
		ts->ts_data.prev_total_num = 0;
	} else if (ts->ts_data.total_num <= ts_caps->max_id) {
		cancel_delayed_work_sync(&ts->work_touch_lock);

		if (ts->gf_ctrl.stage == GHOST_STAGE_CLEAR
				|| (ts->gf_ctrl.stage | GHOST_STAGE_1)
				|| ts->gf_ctrl.stage == GHOST_STAGE_4) {
			ts->ts_data.state = TOUCH_BUTTON_LOCK;
		}

		/* key button cancel */
		if (ts->ts_data.prev_button.state == BUTTON_PRESSED && ts->ts_data.state == TOUCH_BUTTON_LOCK) {
			input_report_key(ts->input_dev, ts->ts_data.prev_button.key_code, BUTTON_CANCELED);

			if (likely(touch_debug_mask_ & (DEBUG_BUTTON | DEBUG_BASE_INFO)))
				TOUCH_INFO_MSG("KEY[%s:%3d] is canceled\n",
						get_touch_button_string(ts->ts_data.prev_button.key_code),
						ts->ts_data.prev_button.key_code);

			memset(&ts->ts_data.prev_button, 0x0, sizeof(ts->ts_data.prev_button));
		}

		if (likely(touch_debug_mask_ & (DEBUG_BASE_INFO | DEBUG_ABS)))
			check_log_finger_changed(ts, ts->ts_data.total_num);

		ts->ts_data.prev_total_num = ts->ts_data.total_num;

		touch_asb_input_report(ts, FINGER_PRESSED);
		report_enable = 1;

		memcpy(ts->ts_data.prev_data, ts->ts_data.curr_data, sizeof(ts->ts_data.curr_data));
	}

	if (report_enable)
		input_sync(ts->input_dev);

	/* Button handle */
	if (ts->ts_data.state != TOUCH_BUTTON_LOCK) {
		/* do not check when there is no pressed button at error case
		 * 	- if you check it, sometimes touch is locked becuase button pressed via IC error.
		 */

		if (ts->work_sync_err_cnt > 0
				&& ts->ts_data.prev_button.state == BUTTON_RELEASED) {
			/* Do nothing */
		} else {
			report_enable = 0;

			if (unlikely(touch_debug_mask_ & DEBUG_BUTTON))
				TOUCH_INFO_MSG("Cur. button -code: %d state: %d, Prev. button -code: %d state: %d\n",
						ts->ts_data.curr_button.key_code,
						ts->ts_data.curr_button.state,
						ts->ts_data.prev_button.key_code,
						ts->ts_data.prev_button.state);

			if (ts->ts_data.curr_button.state == BUTTON_PRESSED
					&& ts->ts_data.prev_button.state == BUTTON_RELEASED) {
				/* button pressed */
				cancel_delayed_work_sync(&ts->work_touch_lock);

				input_report_key(ts->input_dev, ts->ts_data.curr_button.key_code, BUTTON_PRESSED);

				if (likely(touch_debug_mask_ & (DEBUG_BUTTON | DEBUG_BASE_INFO)))
					TOUCH_INFO_MSG("KEY[%s:%3d] is pressed\n",
							get_touch_button_string(ts->ts_data.curr_button.key_code),
							ts->ts_data.curr_button.key_code);

				memcpy(&ts->ts_data.prev_button, &ts->ts_data.curr_button,
						sizeof(ts->ts_data.curr_button));

				report_enable = 1;
			} else if (ts->ts_data.curr_button.state == BUTTON_PRESSED
					&& ts->ts_data.prev_button.state == BUTTON_PRESSED
					&& ts->ts_data.prev_button.key_code != ts->ts_data.curr_button.key_code) {
				/* exception case - multi press button handle */
				queue_delayed_work(touch_wq, &ts->work_touch_lock, msecs_to_jiffies(200));

				/* release previous pressed button */
				input_report_key(ts->input_dev, ts->ts_data.prev_button.key_code, BUTTON_RELEASED);

				ts->ts_data.prev_button.state = BUTTON_RELEASED;

				if (likely(touch_debug_mask_ & (DEBUG_BUTTON | DEBUG_BASE_INFO)))
					TOUCH_INFO_MSG("KEY[%s:%3d] is released\n",
							get_touch_button_string(ts->ts_data.prev_button.key_code),
							ts->ts_data.prev_button.key_code);

				report_enable = 1;
			} else if (ts->ts_data.curr_button.state == BUTTON_RELEASED /* button released */
					&& ts->ts_data.prev_button.state == BUTTON_PRESSED
					&& ts->ts_data.prev_button.key_code == ts->ts_data.curr_button.key_code) {
				/* button release */
				input_report_key(ts->input_dev, ts->ts_data.prev_button.key_code, BUTTON_RELEASED);

				if (likely(touch_debug_mask_ & (DEBUG_BUTTON | DEBUG_BASE_INFO)))
					TOUCH_INFO_MSG("KEY[%s:%3d] is released\n",
							get_touch_button_string(ts->ts_data.prev_button.key_code),
							ts->ts_data.prev_button.key_code);

				memset(&ts->ts_data.prev_button, 0x0, sizeof(ts->ts_data.prev_button));
				memset(&ts->ts_data.curr_button, 0x0, sizeof(ts->ts_data.curr_button));

				report_enable = 1;
			}

			if (report_enable)
				input_sync(ts->input_dev);
		}
	}

out:
	touch_work_post_proc(ts, WORK_POST_OUT);
	return;

power_off:
	return;

err_out_critical:
	touch_work_post_proc(ts, WORK_POST_ERR_CIRTICAL);
	return;

#ifdef LGE_TOUCH_GHOST_DETECTION
err_out_init:
	touch_work_post_proc(ts, WORK_POST_ERR_CIRTICAL);
	return;
#endif
}

static int touch_request_firmware(struct lge_touch_data *ts)
{
	int ret = 0;
	char *fw_name = ts_pdata->fw_image;

	TOUCH_TRACE_FUNC();

	if (ts->fw_info.path[0])
		fw_name = ts->fw_info.path;

	TOUCH_INFO_MSG("request_firmware() %s \n", fw_name);
	ret = request_firmware((const struct firmware **) (&ts->fw_info.fw), fw_name, &ts->client->dev);
	if (ret != 0) {
		ts->fw_info.fw = NULL;
		TOUCH_ERR_MSG("request_firmware() failed %d\n", ret);
	}

	return ret;
}

static int touch_release_firmware(struct lge_touch_data *ts)
{
	TOUCH_TRACE_FUNC();

	if (ts->fw_info.fw) {
		release_firmware((const struct firmware *) ts->fw_info.fw);
	}

	ts->fw_info.fw = NULL;

	return 0;
}

/* touch_fw_upgrade_func
 *
 * it used to upgrade the firmware of touch IC.
 */
static void touch_fw_upgrade_func(struct work_struct *work_fw_upgrade)
{
	struct lge_touch_data *ts =
			container_of(work_fw_upgrade, struct lge_touch_data, work_fw_upgrade);
	u8	saved_state = ts->curr_pwr_state;

	TOUCH_TRACE_FUNC();

	if (touch_drv->fw_upgrade == NULL) {
		TOUCH_INFO_MSG("There is no specific firmware upgrade function\n");
		goto out;
	}

	ts->fw_info.is_downloading = UNDER_DOWNLOADING;
	power_lock(POWER_FW_UP_LOCK);

	if (touch_request_firmware(ts)) {
		TOUCH_INFO_MSG("FW-upgrade is not excuted\n");
		goto out;
	}

	if (ts->curr_pwr_state == POWER_ON || ts->curr_pwr_state == POWER_WAKE) {
		touch_disable(ts);
	}

#ifdef LGE_TOUCH_GHOST_DETECTION
	if (ts_role->ghost_detection_enable) {
		hrtimer_cancel(&ts->trigger_timer);
	}
#endif

	if (ts->curr_pwr_state == POWER_OFF) {
		touch_power_cntl(ts, POWER_ON);
		msleep(ts_role->booting_delay);
	}

	if (likely(touch_debug_mask_ & (DEBUG_FW_UPGRADE | DEBUG_BASE_INFO)))
		TOUCH_INFO_MSG("F/W upgrade - Start\n");

#ifdef LGE_TOUCH_TIME_DEBUG
	do_gettimeofday(&t_debug[TIME_FW_UPGRADE_START]);
#endif
	if (touch_drv->fw_upgrade(ts->client, &ts->fw_info) < 0) {
		TOUCH_ERR_MSG("Firmware upgrade was failed\n");
		touch_enable(ts);
		goto err_out;
	}

#ifdef LGE_TOUCH_TIME_DEBUG
	do_gettimeofday(&t_debug[TIME_FW_UPGRADE_END]);
#endif

	touch_power_cntl(ts, POWER_OFF);

	touch_power_cntl(ts, POWER_ON);
	msleep(ts_role->booting_delay);
	touch_ic_init(ts);

	touch_drv->ic_ctrl(ts->client, IC_CTRL_SAVE_IC_INFO, 0);
	TOUCH_INFO_MSG("power saved_state : %d \n", saved_state);
	if (saved_state != POWER_ON)
		touch_power_cntl(ts, saved_state);
	else
		touch_enable(ts);


	if (likely(touch_debug_mask_ & (DEBUG_FW_UPGRADE | DEBUG_BASE_INFO)))
		TOUCH_INFO_MSG("F/W upgrade - Finish\n");

#ifdef LGE_TOUCH_TIME_DEBUG
	do_gettimeofday(&t_debug[TIME_FW_UPGRADE_END]);

	if (touch_time_debug_mask & DEBUG_TIME_FW_UPGRADE
			|| touch_time_debug_mask & DEBUG_TIME_PROFILE_ALL) {
		TOUCH_INFO_MSG("FW upgrade time is under %3lu.%06lusec\n",
				get_time_interval(t_debug[TIME_FW_UPGRADE_END].tv_sec, t_debug[TIME_FW_UPGRADE_START].tv_sec),
				get_time_interval(t_debug[TIME_FW_UPGRADE_END].tv_usec, t_debug[TIME_FW_UPGRADE_START].tv_usec));
	}
#endif

	goto out;

err_out:
	power_unlock(POWER_FW_UP_LOCK);
	safety_reset(ts);
	touch_ic_init(ts);

out:
	/* Specific device resolution */
	touch_release_firmware(ts);

	if (touch_drv->resolution) {
		if (touch_drv->resolution(ts->client) < 0) {
			TOUCH_ERR_MSG("specific device resolution fail\n");
		}
	}
	input_set_abs_params(ts->input_dev, ABS_MT_POSITION_X, 0, ts_caps->x_max, 0, 0);
	input_set_abs_params(ts->input_dev, ABS_MT_POSITION_Y, 0, ts_caps->y_max, 0, 0);
	ts->fw_info.is_downloading = DOWNLOAD_COMPLETE;
	power_unlock(POWER_FW_UP_LOCK);
	return;
}

/* touch_irq_handler
 *
 * When Interrupt occurs, it will be called before touch_thread_irq_handler.
 *
 * return
 * IRQ_HANDLED: touch_thread_irq_handler will not be called.
 * IRQ_WAKE_THREAD: touch_thread_irq_handler will be called.
 */
static irqreturn_t touch_irq_handler(int irq, void *dev_id)
{
	struct lge_touch_data *ts = (struct lge_touch_data *)dev_id;

	if (unlikely(atomic_read(&ts->device_init) != 1))
		return IRQ_HANDLED;

#ifdef LGE_TOUCH_TIME_DEBUG
	do_gettimeofday(&t_debug[TIME_ISR_START]);
#endif

	return IRQ_WAKE_THREAD;
}

/* touch_thread_irq_handler
 *
 * 1. disable irq.
 * 2. enqueue the new work.
 * 3. enalbe irq.
 */
static irqreturn_t touch_thread_irq_handler(int irq, void *dev_id)
{
	struct lge_touch_data *ts = (struct lge_touch_data *)dev_id;

#ifdef LGE_TOUCH_TIME_DEBUG
	do_gettimeofday(&t_debug[TIME_THREAD_ISR_START]);
#endif

	INIT_COMPLETION(ts->irq_completion);

	touch_work_func_a(ts);

	complete_all(&ts->irq_completion);

	return IRQ_HANDLED;
}


/* check_platform_data
 *
 * check-list
 * 1. Null Pointer
 * 2. lcd, touch screen size
 * 3. button support
 * 4. operation mode
 * 5. power module
 * 6. report period
 */
static int check_platform_data(struct i2c_client *client)
{
	int i;
	struct lge_touch_data *ts = i2c_get_clientdata(client);

	TOUCH_TRACE_FUNC();

	if (!ts_pdata)
		return -1;

	if (!ts_caps || !ts_role)
		return -1;

	if (!ts_caps->lcd_x || !ts_caps->lcd_y || !ts_caps->x_max || !ts_caps->y_max) {
		TOUCH_ERR_MSG("lcd_x, lcd_y, x_max, y_max are should be defined\n");
		return -1;
	}

	if (ts_caps->button_support) {
		if (!ts_role->key_type) {
			TOUCH_ERR_MSG("button_support = 1, but key_type is not defined\n");
			return -1;
		}

		if (!ts_caps->y_button_boundary) {
			if (ts_role->key_type == TOUCH_HARD_KEY)
				ts_caps->y_button_boundary = ts_caps->y_max;
			else
				ts_caps->y_button_boundary =
					(ts_caps->lcd_y * ts_caps->x_max) / ts_caps->lcd_x;
		}

		if (ts_caps->button_margin < 0 || ts_caps->button_margin > 49) {
			ts_caps->button_margin = 10;
			TOUCH_ERR_MSG("0 < button_margin < 49, button_margin is set 10 by force\n");
		}
	}

		if (!gpio_is_valid(ts_pdata->int_pin)) {
			TOUCH_ERR_MSG("gpio must be set for interrupt mode\n");
			return -1;
		}

		if (client->irq != gpio_to_irq(ts_pdata->int_pin)) {
			TOUCH_ERR_MSG("warning!! irq[%d] and client->irq[%d] are different\n",
					gpio_to_irq(ts_pdata->int_pin), client->irq);
			client->irq = gpio_to_irq(ts_pdata->int_pin);
		}

	if (ts_role->suspend_pwr == POWER_OFF)
		ts_role->resume_pwr = POWER_ON;
	else if (ts_role->suspend_pwr == POWER_SLEEP)
		ts_role->resume_pwr = POWER_WAKE;
	else
		TOUCH_ERR_MSG("suspend_pwr = POWER_OFF or POWER_SLEEP\n");

	for (i = 0; i < TOUCH_PWR_NUM; ++i) {
		if (ts_pwr[i].type == 1 && !gpio_is_valid(ts_pwr[i].value)) {
			TOUCH_ERR_MSG("you should assign gpio for pwr[%d]\n", i);
			return -1;
		} else if (ts_pwr[i].type == 2) {
			if (!ts_pwr[i].name[0]) {
				TOUCH_ERR_MSG("you should assign the supply name for regulator[%d]\n", i);
				return -1;
			}
		}
	}


	if (ts_caps->max_id > MAX_FINGER)
		ts_caps->max_id = MAX_FINGER;

	return 0;
}


enum touch_dt_type {
	DT_U8,
	DT_U16,
	DT_U32,
	DT_GPIO,
	DT_STRING,
};

struct touch_dt_map {
	const char		*name;
	void			*data;
	enum touch_dt_type	type;
	int			def_value;
};

static char *touch_print_dt_type(enum touch_dt_type type)
{
	static char *str[10] = {
		"DT_U8",
		"DT_U16",
		"DT_U32",
		"DT_GPIO",
		"DT_STRING",
	};

	return str[type];
}

static int touch_parse_dt(struct device *dev, struct touch_platform_data *pdata)
{
	struct device_node *np = dev->of_node;
	struct property *prop;

	struct touch_device_caps *caps = pdata->caps;
	struct touch_operation_role *role = pdata->role;
	struct touch_power_info *pwr = pdata->pwr;

	int i, ret;
	char *tmp;
	u32 val;
	const __be32 *value;
	const char *str_index = NULL;
	struct touch_dt_map *itr;
	struct touch_dt_map map[] = {
		{ "lge,sda-gpio", &pdata->sda_pin, DT_GPIO, 0xffffffff },
		{ "lge,scl-gpio", &pdata->scl_pin, DT_GPIO, 0xffffffff },
		{ "lge,int-gpio", &pdata->int_pin, DT_GPIO, 0xffffffff },
		{ "lge,id-gpio", &pdata->id_pin, DT_GPIO, 0 },
		{ "lge,id2-gpio", &pdata->id2_pin, DT_GPIO, 0 },
		{ "lge,rst-gpio", &pdata->reset_pin, DT_GPIO, 0xffffffff },

		{ "lge,ic_type", &pdata->ic_type, DT_U8, 0 },
		{ "lge,maker", &pdata->maker, DT_STRING, 0 },
		{ "lge,product", &pdata->fw_product, DT_STRING, 0 },
		{ "lge,fw_image", &pdata->fw_image, DT_STRING, 0 },
		{ "lge,auto_fw_update", &pdata->auto_fw_update, DT_U8, 0 },
		{ "lge,panel_spec", &pdata->panel_spec, DT_STRING, 0 },

		/* caps */
		{ "button_support", &caps->button_support, DT_U8, 0 },
		{ "y_button_boundary", &caps->y_button_boundary, DT_U16, 0 },
		{ "button_margin", &caps->button_margin, DT_U32, 0 },

		/* { "number_of_button", &caps->number_of_button, DT_U8, 0 }, */
		{ "is_width_supported", &caps->is_width_supported, DT_U8, 0 },
		{ "is_pressure_supported", &caps->is_pressure_supported, DT_U8, 0 },
		{ "is_id_supported", &caps->is_id_supported, DT_U8, 0 },
		{ "max_width", &caps->max_width, DT_U32, 0 },
		{ "max_pressure", &caps->max_pressure, DT_U32, 0 },
		{ "max_id", &caps->max_id, DT_U32, 0 },
		{ "x_max", &caps->x_max, DT_U32, 0 },
		{ "y_max", &caps->y_max, DT_U32, 0 },
		{ "lcd_x", &caps->lcd_x, DT_U32, 0 },
		{ "lcd_y", &caps->lcd_y, DT_U32, 0 },

		/* role */
		{ "key_type", &role->key_type, DT_U8, 0 },
		{ "booting_delay", &role->booting_delay, DT_U32, 0 },
		{ "reset_delay", &role->reset_delay, DT_U32, 0 },
		{ "suspend_pwr", &role->suspend_pwr, DT_U8, 0 },
		{ "resume_pwr", &role->resume_pwr, DT_U8, 0 },
		{ "ghost_finger_solution_enable", &role->ghost_finger_solution_enable, DT_U32, 0 },
		{ "ghost_detection_enable", &role->ghost_finger_solution_enable, DT_U32, 0 },
		/* { "irqflags", &role->irqflags, DT_U32, 0 }, */

		/* power_info */
		{ "vdd_type0", &pwr[0].type, DT_U8, 0 },
		{ "vdd_name0", &pwr[0].name, DT_STRING, 0 },
		{ "vdd_value0", &pwr[0].value, DT_U32, 0xffffffff },
		{ "vdd_type1", &pwr[1].type, DT_U8, 0 },
		{ "vdd_name1", &pwr[1].name, DT_STRING, 0 },
		{ "vdd_value1", &pwr[1].value, DT_U32, 0xffffffff },
		{ "vdd_type2", &pwr[2].type, DT_U8, 0 },
		{ "vdd_name2", &pwr[2].name, DT_STRING, 0 },
		{ "vdd_value2", &pwr[2].value, DT_U32, 0xffffffff },
		{ NULL, NULL, 0, 0},
	};

	TOUCH_TRACE_FUNC();

	/* reset, irq gpio info */
	if (np == NULL)
		return -ENODEV;

	for (itr = map; itr->name; ++itr) {
		val = 0;
		ret = 0;

		switch (itr->type) {
		case DT_U8:
			ret = of_property_read_u32(np, itr->name, &val);
			if (ret == 0) {
				*((u8 *) itr->data) = (u8) val;
				if (unlikely(touch_debug_mask_ & DEBUG_CONFIG))
					TOUCH_DEBUG_MSG("DT entry: %s = %d (%s)\n", itr->name, val, touch_print_dt_type(itr->type));
			} else {
				*((u8 *) itr->data) = (u8) itr->def_value;
			}

			break;

		case DT_U16:
			ret = of_property_read_u32(np, itr->name, &val);
			if (ret == 0) {
				*((u16 *) itr->data) = (u16) val;
				if (unlikely(touch_debug_mask_ & DEBUG_CONFIG))
					TOUCH_DEBUG_MSG("DT entry: %s = %d (%s)\n", itr->name, val, touch_print_dt_type(itr->type));
			} else {
				*((u16 *) itr->data) = (u16) itr->def_value;
			}
			break;

		case DT_U32:
			ret = of_property_read_u32(np, itr->name, &val);
			if (ret == 0) {
				*((u32 *) itr->data) = val;
				if (unlikely(touch_debug_mask_ & DEBUG_CONFIG))
					TOUCH_DEBUG_MSG("DT entry: %s = %d (%s)\n", itr->name, val, touch_print_dt_type(itr->type));
			} else {
				*((u32 *) itr->data) = (u32) itr->def_value;
			}
			break;

		case DT_STRING:
			tmp = NULL;
			ret = of_property_read_string(np, itr->name, (const char **) &tmp);
			if (ret == 0) {
				strcpy(itr->data, tmp);
				if (unlikely(touch_debug_mask_ & DEBUG_CONFIG))
					TOUCH_DEBUG_MSG("DT entry: %s = %s (%s)\n", itr->name, (char *) itr->data, touch_print_dt_type(itr->type));
			} else {
				*((u32 *) itr->data) = 0;
			}
			break;

		case DT_GPIO:
			ret = of_get_named_gpio(np, itr->name, 0);
			if (ret >= 0) {
				if (unlikely(touch_debug_mask_ & DEBUG_CONFIG))
					TOUCH_DEBUG_MSG("DT entry: %s = %d (%s)\n", itr->name, ret, touch_print_dt_type(itr->type));
				*((int *) itr->data) = (int) ret;
				ret = 0;
			} else {
				*((int *) itr->data) = (int) -1;
			}

			break;

		default:
			ret = -EBADE;
		}

		if (ret) {
			if (unlikely(touch_debug_mask_ & DEBUG_CONFIG))
				TOUCH_DEBUG_MSG("Missing DT entry: %s, ret:%d", itr->name, ret);
		}
	}

	if (role->key_type == 0)
		role->key_type = TOUCH_HARD_KEY;

	prop = of_find_property(np, "button_name", &i);
	if (prop == NULL) {
		if (unlikely(touch_debug_mask_ & DEBUG_CONFIG))
			TOUCH_DEBUG_MSG("Missing DT entry: %s, ret:%d", itr->name, ret);
		return 0;
	}

	caps->number_of_button = (i >> 2);

	if (caps->number_of_button > MAX_BUTTON) {
		if (unlikely(touch_debug_mask_ & DEBUG_CONFIG))
			TOUCH_DEBUG_MSG("too many buttons : %d !!\n", caps->number_of_button);
		caps->number_of_button = MAX_BUTTON;
	}

	if (unlikely(touch_debug_mask_ & DEBUG_CONFIG))
		TOUCH_DEBUG_MSG("DT entry: number_of_button = %d\n", caps->number_of_button);

	value = prop->value;

	for (i = 0; i < caps->number_of_button; i++) {
		caps->button_name[i] = be32_to_cpup(value++);

		if (unlikely(touch_debug_mask_ & DEBUG_CONFIG))
			TOUCH_DEBUG_MSG("DT entry: button_name[%d] : %s:%d\n", i,
				get_touch_button_string((u16)caps->button_name[i]), caps->button_name[i]);
	}

	ret = of_property_count_strings(np, "panel_id");
	if (ret) {
		for (i = 0; i < ret; i++) {
			of_property_read_string_index(np, "panel_id", i, &str_index);
			pdata->panel_type[i] = str_index[0];
		}
	}

	return 0;
}

static int touch_init_platform_data(struct i2c_client *client)
{
	struct lge_touch_data *ts = i2c_get_clientdata(client);
	struct section_info *sc = &ts->st_info;

	int one_sec = 0;
	int i;

	if (client->dev.of_node) {
		ts_pdata = devm_kzalloc(&client->dev,
				sizeof(struct touch_platform_data), GFP_KERNEL);
		if (!ts_pdata) {
			dev_err(&client->dev, "Failed to allocate memory : pdata\n");
			return -ENOMEM;
		}

		ts_caps = devm_kzalloc(&client->dev,
				sizeof(struct touch_device_caps), GFP_KERNEL);
		if (!ts_caps) {
			dev_err(&client->dev, "Failed to allocate memory : caps\n");
			return -ENOMEM;
		}

		ts_role = devm_kzalloc(&client->dev,
				sizeof(struct touch_operation_role), GFP_KERNEL);
		if (!ts_role) {
			dev_err(&client->dev, "Failed to allocate memory : role\n");
			return -ENOMEM;
		}

		if (touch_parse_dt(&client->dev, ts_pdata) < 0)
			return -EINVAL;
	} else {
		ts_pdata = client->dev.platform_data;
		if (!ts_pdata) {
			return -EINVAL;
		}
	}

	if (check_platform_data(client) < 0) {
		TOUCH_ERR_MSG("platform data check failed\n");
		return -EINVAL;
	}
	one_sec = 1000000 / (12500000 /*ts_role->report_period*/ / 1000);
	ts->ic_init_err_cnt = 0;
	ts->work_sync_err_cnt = 0;

	/* Ghost-Finger solution
	 * max_count : if melt-time > 7-sec, max_count should be changed to 'one_sec * 3'
	 */
	ts->gf_ctrl.min_count = one_sec / 12;
	ts->gf_ctrl.max_count = one_sec * 3;
	ts->gf_ctrl.saved_x = -1;
	ts->gf_ctrl.saved_y = -1;
	ts->gf_ctrl.saved_last_x = 0;
	ts->gf_ctrl.saved_last_y = 0;
	ts->gf_ctrl.max_moved = ts_caps->x_max / 4;
	ts->gf_ctrl.max_pressure = 255;

	sc->panel.left = 0;
	sc->panel.right = ts_caps->x_max;
	sc->panel.top = 0;
	sc->panel.bottom = ts_caps->y_button_boundary;

	sc->b_width  = ts_caps->x_max / ts_caps->number_of_button;
	sc->b_margin = sc->b_width * ts_caps->button_margin / 100;
	sc->b_inner_width = sc->b_width - (2*sc->b_margin);
	sc->b_height = ts_caps->y_max - ts_caps->y_button_boundary;
	sc->b_num = ts_caps->number_of_button;

	for (i = 0; i < sc->b_num; i++) {
		sc->button[i].left   = i * (ts_caps->x_max / ts_caps->number_of_button) + sc->b_margin;
		sc->button[i].right  = sc->button[i].left + sc->b_inner_width;
		sc->button[i].top    = ts_caps->y_button_boundary + 1;
		sc->button[i].bottom = ts_caps->y_max;

		sc->button_cancel[i].left = sc->button[i].left - (2*sc->b_margin) >= 0 ?
			sc->button[i].left - (2*sc->b_margin) : 0;
		sc->button_cancel[i].right = sc->button[i].right + (2*sc->b_margin) <= ts_caps->x_max ?
			sc->button[i].right + (2*sc->b_margin) : ts_caps->x_max;
		sc->button_cancel[i].top = sc->button[i].top;
		sc->button_cancel[i].bottom = sc->button[i].bottom;

		sc->b_name[i] = ts_caps->button_name[i];
	}

	return 0;
}


static int touch_input_init(struct lge_touch_data *ts)
{
	struct input_dev *dev;
	int i;
	int max;

	dev = input_allocate_device();

	if (dev == NULL) {
		TOUCH_ERR_MSG("Failed to allocate input device\n");
		return -ENOMEM;
	}

	dev->name = "touch_dev";
	dev->dev.init_name = LGE_TOUCH_NAME;


	set_bit(EV_SYN, dev->evbit);
	set_bit(EV_ABS, dev->evbit);
	set_bit(INPUT_PROP_DIRECT, dev->propbit);

	if (ts_caps->button_support && ts_role->key_type != VIRTUAL_KEY) {
		set_bit(EV_KEY, dev->evbit);
		for (i = 0; i < ts_caps->number_of_button; i++) {
			set_bit(ts_caps->button_name[i], dev->keybit);
		}
	}

	input_set_abs_params(dev, ABS_MT_POSITION_X, 0, ts_caps->x_max, 0, 0);
	max = (ts_caps->y_button_boundary ? ts_caps->y_button_boundary : ts_caps->y_max);
	input_set_abs_params(dev, ABS_MT_POSITION_Y, 0, max, 0, 0);

	if (ts_caps->is_pressure_supported)
		input_set_abs_params(dev, ABS_MT_PRESSURE, 0, ts_caps->max_pressure, 0, 0);

	if (ts_caps->is_width_supported) {
		input_set_abs_params(dev, ABS_MT_WIDTH_MAJOR, 0, ts_caps->max_width, 0, 0);
		input_set_abs_params(dev, ABS_MT_WIDTH_MINOR, 0, ts_caps->max_width, 0, 0);
		input_set_abs_params(dev, ABS_MT_ORIENTATION, 0, 1, 0, 0);
	}

	input_mt_init_slots(dev, ts_caps->max_id);

	if (input_register_device(dev) < 0) {
		TOUCH_ERR_MSG("Unable to register %s input device\n",
				dev->name);
		input_free_device(dev);
		return -EINVAL;
	}

	input_set_drvdata(dev, ts);
	ts->input_dev = dev;

	return 0;
}

static int touch_suspend(struct device *device)
{
	struct lge_touch_data *ts =  dev_get_drvdata(device);
	long ret = 0;

	TOUCH_TRACE_FUNC();

	ts->pdata->panel_on = 0;
	if (ts->curr_pwr_state == POWER_OFF)
		return 0;

	if (power_block) {
		TOUCH_INFO_MSG("touch_suspend is not executed\n");
		return 0;
	}

	touch_disable(ts);

	cancel_delayed_work_sync(&ts->work_init);
	if (ts_role->key_type == TOUCH_HARD_KEY)
		cancel_delayed_work_sync(&ts->work_touch_lock);

	ret = wait_for_completion_interruptible_timeout(&ts->irq_completion, msecs_to_jiffies(1000));
	if (ret == 0) {
		TOUCH_INFO_MSG("Completion timeout \n");
	} else if (ret < 0) {
		TOUCH_INFO_MSG("Completion ret = %ld \n", ret);
	}

	release_all_ts_event(ts);

	if (ts->ts_data.palm) {
		TOUCH_INFO_MSG("Palm released \n");
	}

	memset(&ts->ts_data, 0x0, sizeof(ts->ts_data));

	touch_power_cntl(ts, ts_role->suspend_pwr);

	return 0;
}

static int touch_resume(struct device *device)
{
	struct lge_touch_data *ts =  dev_get_drvdata(device);

	TOUCH_TRACE_FUNC();

	ts->pdata->panel_on = 1;
	if (ts->curr_pwr_state == POWER_ON)
		return 0;

	if (power_block) {
		TOUCH_INFO_MSG("touch_resume is not executed\n");
		return 0;
	}

	touch_power_cntl(ts, ts_role->resume_pwr);
	touch_enable(ts);
	ts->ic_error_cnt = 0;

	if (ts_pdata->role->resume_pwr == POWER_ON)
		queue_delayed_work(touch_wq, &ts->work_init,
				msecs_to_jiffies(ts_role->booting_delay));
	else
		queue_delayed_work(touch_wq, &ts->work_init, 0);

	return 0;
}

#if defined (CONFIG_HAS_EARLYSUSPEND)
static void touch_early_suspend(struct early_suspend *h)
{
	struct lge_touch_data *ts =
		container_of(h, struct lge_touch_data, early_suspend);

	TOUCH_TRACE_FUNC();

	touch_suspend(&ts->client->dev);
}

static void touch_late_resume(struct early_suspend *h)
{
	struct lge_touch_data *ts =
		container_of(h, struct lge_touch_data, early_suspend);

	TOUCH_TRACE_FUNC();

	touch_suspend(&ts->client->dev);
}
#endif


#if defined(CONFIG_FB)
static int touch_notifier_callback(struct notifier_block *self, unsigned long event, void *data)
{
	struct fb_event *evdata = data;
	int *blank;
	struct lge_touch_data *ts = container_of(self, struct lge_touch_data, fb_notifier_block);

	if (evdata && evdata->data && event == FB_EVENT_BLANK && ts && ts->client) {
		blank = evdata->data;
		if (*blank == FB_BLANK_UNBLANK) {
			touch_resume(&ts->input_dev->dev);
		} else if (*blank == FB_BLANK_POWERDOWN) {
			touch_suspend(&ts->input_dev->dev);
		}
	}

	return 0;
}
#endif

/* sysfs */
static ssize_t platform_data_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct lge_touch_data *ts = dev_get_drvdata(dev);
	int i;
	int ret = 0;

	ret += sprintf(buf+ret, "====== Platform data ======\n");
	ret += sprintf(buf+ret, "int_pin[%d] reset_pin[%d]\n", ts_pdata->int_pin, ts_pdata->reset_pin);
	ret += sprintf(buf+ret, "scl_pin[%d] sda_pin[%d]\n", ts_pdata->scl_pin, ts_pdata->sda_pin);
	if (ts_pdata->id_pin)
		ret += sprintf(buf+ret, "Touch ID[%d]\n", ts_pdata->id_pin);
	if (ts_pdata->id_pin)
		ret += sprintf(buf+ret, "Touch ID2[%d]\n", ts_pdata->id2_pin);
	ret += sprintf(buf+ret, "maker [%s %s] \n", ts_pdata->maker, ts_pdata->ic_type ? MMS_100A : MMS_100S);
	ret += sprintf(buf+ret, "defualt firmware image[%s]\n", ts_pdata->fw_image);
	ret += sprintf(buf+ret, "default panel spec[%s]\n", ts_pdata->panel_spec);

	ret += sprintf(buf+ret, "caps:\n");
	ret += sprintf(buf+ret, "\tbutton_support        = %d\n", ts_caps->button_support);
	ret += sprintf(buf+ret, "\ty_button_boundary     = %d\n", ts_caps->y_button_boundary);
	ret += sprintf(buf+ret, "\tbutton_margin         = %d\n", ts_caps->button_margin);
	ret += sprintf(buf+ret, "\tnumber_of_button      = %d\n", ts_caps->number_of_button);
	ret += sprintf(buf+ret, "\tbutton_name           = %d, %d, %d, %d\n",
			ts_caps->button_name[0], ts_caps->button_name[1], ts_caps->button_name[2], ts_caps->button_name[3]);
	ret += sprintf(buf+ret, "\tis_width_supported    = %d\n", ts_caps->is_width_supported);
	ret += sprintf(buf+ret, "\tis_pressure_supported = %d\n", ts_caps->is_pressure_supported);
	ret += sprintf(buf+ret, "\tis_id_supported       = %d\n", ts_caps->is_id_supported);
	ret += sprintf(buf+ret, "\tmax_width             = %d\n", ts_caps->max_width);
	ret += sprintf(buf+ret, "\tmax_pressure          = %d\n", ts_caps->max_pressure);
	ret += sprintf(buf+ret, "\tmax_id                = %d\n", ts_caps->max_id);
	ret += sprintf(buf+ret, "\tx_max                 = %d\n", ts_caps->x_max);
	ret += sprintf(buf+ret, "\ty_max                 = %d\n", ts_caps->y_max);
	ret += sprintf(buf+ret, "\tlcd_x                 = %d\n", ts_caps->lcd_x);
	ret += sprintf(buf+ret, "\tlcd_y                 = %d\n", ts_caps->lcd_y);

	ret += sprintf(buf+ret, "role:\n");
	ret += sprintf(buf+ret, "\tbooting_delay         = %d\n", ts_role->booting_delay);
	ret += sprintf(buf+ret, "\treset_delay           = %d\n", ts_role->reset_delay);
	ret += sprintf(buf+ret, "\tsuspend_pwr           = %d\n", ts_role->suspend_pwr);
	ret += sprintf(buf+ret, "\tresume_pwr            = %d\n", ts_role->resume_pwr);
	ret += sprintf(buf+ret, "\tghost_finger_solution_enable = %d\n", ts_role->ghost_finger_solution_enable);
	/* ret += sprintf(buf+ret, "\tirqflags              = 0x%lx\n", ts_role->irqflags); */
#ifdef LGE_TOUCH_GHOST_DETECTION
	ret += sprintf(buf+ret, "\tghost_detection_enable= %d\n", ts_role->ghost_detection_enable);
#endif
	ret += sprintf(buf+ret, "pwr:\n");
	for (i = 0; i < TOUCH_PWR_NUM; i++) {
		ret += sprintf(buf+ret, "\t[%d] vdd_type         = %d\n", i, ts_pwr[i].type);
		ret += sprintf(buf+ret, "\t[%d] name             = %s\n", i, ts_pwr[i].name);
		ret += sprintf(buf+ret, "\t[%d] value            = %d\n", i, ts_pwr[i].value);
	}
	return ret;
}

static ssize_t fw_upgrade_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct lge_touch_data *ts = dev_get_drvdata(dev);
	int ret = 0;

	ret += sprintf(buf + ret, "\n");
	ret += sprintf(buf + ret, "help for fw_upgrade\n");
	ret += sprintf(buf + ret, "   echo [option] > fw_upgrade\n");
	ret += sprintf(buf + ret, "\n");
	ret += sprintf(buf + ret, "option:\n");

	if (ts_pdata->ic_type)
		ret += sprintf(buf + ret, "  0 : Upgrade default Core54 for MMS-100A \n");
	else
		ret += sprintf(buf + ret, "  0 : Upgrade default Core32 for MMS-100S \n");
	ret += sprintf(buf + ret, "  1 : Upgrade with the given name in Device Tree [%s]\n", ts_pdata->fw_image);
	ret += sprintf(buf + ret, "  9 : Upgrade with %s%s\n", EXTERNAL_FW_PATH, EXTERNAL_FW_NAME);
	ret += sprintf(buf + ret, "\n");
	ret += sprintf(buf + ret, "cat fw_upgrade\n");
	ret += sprintf(buf + ret, "    : show this message\n");

	return ret;
}

static ssize_t fw_upgrade_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct lge_touch_data *ts = dev_get_drvdata(dev);
	int cmd = 0;

	if (sscanf(buf, "%d", &cmd) != 1)
		return -EINVAL;

	switch (cmd) {
	case 0:
		if (ts_pdata->ic_type)
			strncpy(ts->fw_info.path, CORE54_FW_NAME, NAME_BUFFER_SIZE);
		else
			strncpy(ts->fw_info.path, CORE32_FW_NAME, NAME_BUFFER_SIZE);
		break;

	case 1:
		strncpy(ts->fw_info.path, ts_pdata->fw_image, NAME_BUFFER_SIZE);
		break;

	case 9:
		snprintf(ts->fw_info.path, NAME_BUFFER_SIZE, "%s", EXTERNAL_FW_NAME);
		break;

	default:
		ts->fw_info.path[0] = '\0';
		return count;
	}

	while (ts->fw_info.is_downloading)
	;

	ts->fw_info.is_downloading = 1;
	ts->fw_info.force_upgrade = 1;

	queue_work(touch_wq, &ts->work_fw_upgrade);

	while (ts->fw_info.is_downloading)
	;

	return count;
}

static ssize_t firmware_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	int ret = 0;
	struct ic_ctrl_param param;
	struct lge_touch_data *ts = dev_get_drvdata(dev);

	if (touch_request_firmware(ts) < 0) {
		ret += sprintf(buf + ret, "firmware request failed\n");
	} else {
		param.v1 = (u32) ts->fw_info.fw->data;
		param.v2 = (u32) buf;
		param.v3 = (u32) ret;
		ret = touch_drv->ic_ctrl(ts->client, IC_CTRL_FIRMWARE_IMG_SHOW, (u32) &param);
		touch_release_firmware(ts);
	}

	ret += sprintf(buf + ret, "\n");
	ret += sprintf(buf + ret, "help for firmware\n");
	ret += sprintf(buf + ret, "   echo [option] [filename] > firmware\n");
	ret += sprintf(buf + ret, "\n");
	ret += sprintf(buf + ret, "option:\n");
	ret += sprintf(buf + ret, "  -s : just file name update. no upgrade\n");
	ret += sprintf(buf + ret, "  -u : upgrade with the given file\n");
	ret += sprintf(buf + ret, "       if no filename is given, upgrades to the default.\n");
	ret += sprintf(buf + ret, "  -f : force upgrade with the given file\n");
	ret += sprintf(buf + ret, "       if no filename is given, upgrades to the default.\n");
	ret += sprintf(buf + ret, "\n");
	ret += sprintf(buf + ret, "   cat firmware\n");
	ret += sprintf(buf + ret, "     : show this message\n");

	return ret;
}

static ssize_t firmware_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct lge_touch_data *ts = dev_get_drvdata(dev);
	int i;
	char opt = 0;

	if (count < 3)
		return -EINVAL;

	if (buf[0] != '-')
		return -EINVAL;

	if (buf[1] != '\n')
		opt = buf[1];

	if (opt != 's' && opt != 'u' && opt != 'f') {
		TOUCH_INFO_MSG("unknow option = %c\n", opt);
		return -EINVAL;
	}

	while (ts->fw_info.is_downloading)
	;

	ts->fw_info.path[0] = '\0';

	if (buf[2] == ' ' && count < (NAME_BUFFER_SIZE + 3)) {
		for (i = 3; i < count; ++i) {
			if (buf[i] == '\n' || buf[i] == ' ') {
				ts->fw_info.path[i-3] = '\0';
				break;
			} else {
				ts->fw_info.path[i-3] = buf[i];
			}
		}
	}

	TOUCH_INFO_MSG("%s : buf[%s], count[%d]\n", __func__, buf, count);
	TOUCH_INFO_MSG("%s : opt[%c], path[%s]\n", __func__, opt, ts->fw_info.path);

	if (opt == 's')
		goto sysfs_firmware_store_exit;

	while (ts->fw_info.is_downloading)
	;

	ts->fw_info.is_downloading = 1;
	ts->fw_info.force_upgrade = (opt == 'f' ? true : false);

	queue_work(touch_wq, &ts->work_fw_upgrade);

	while (ts->fw_info.is_downloading)
	;

sysfs_firmware_store_exit:
	return count;
}

static ssize_t version_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct lge_touch_data *ts = dev_get_drvdata(dev);
	int ret = 0;

	ret = touch_drv->sysfs(ts->client, buf, 0, SYSFS_VERSION_SHOW);
	return ret;
}

static ssize_t testmode_version_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct lge_touch_data *ts = dev_get_drvdata(dev);
	int ret = 0;

	ret = touch_drv->sysfs(ts->client, buf, 0, SYSFS_TESTMODE_VERSION_SHOW);
	return ret;
}

static ssize_t status_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct lge_touch_data *ts = dev_get_drvdata(dev);
	int len = 0;
	len = snprintf(buf, PAGE_SIZE, "\nLGE-Touch Device Status\n");
	len += snprintf(buf + len, PAGE_SIZE - len, "=============================\n");
	len += snprintf(buf + len, PAGE_SIZE - len, "irq num       is %d\n", ts->client->irq);
	if (ts->pdata->panel_id != 0xFF)
		len += snprintf(buf + len, PAGE_SIZE - len, "panel id      is %d [%c]\n", ts_pdata->panel_id, ts_pdata->panel_type[ts_pdata->panel_id]);
	len += snprintf(buf + len, PAGE_SIZE - len, "gpio_irq num  is %d (level=%d)\n", ts_pdata->int_pin, gpio_get_value(ts_pdata->int_pin));
	len += snprintf(buf + len, PAGE_SIZE - len, "gpio_scl num  is %d\n", ts_pdata->scl_pin);
	len += snprintf(buf + len, PAGE_SIZE - len, "gpio_sda num  is %d\n", ts_pdata->sda_pin);
	if (ts_pdata->id_pin)
		len += snprintf(buf + len, PAGE_SIZE - len, "gpio_id num   is %d (level=%d)\n", ts_pdata->id_pin, gpio_get_value(ts_pdata->id_pin));
	if (ts_pdata->id2_pin)
		len += snprintf(buf + len, PAGE_SIZE - len, "gpio_id2 num  is %d (level=%d)\n", ts_pdata->id2_pin, gpio_get_value(ts_pdata->id2_pin));

	return len;
}

static ssize_t power_control_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct lge_touch_data *ts = dev_get_drvdata(dev);

	int cmd = 0;

	if (sscanf(buf, "%d", &cmd) != 1)
		return -EINVAL;
	switch (cmd) {
	case 0:
		if (ts->curr_pwr_state == POWER_ON) {
			touch_disable(ts);
			touch_power_cntl(ts, POWER_OFF);
		}
		break;
	case 1:
		if (ts->curr_pwr_state == POWER_OFF) {
			touch_power_cntl(ts, POWER_ON);
			msleep(ts_role->booting_delay);
			touch_enable(ts);
			touch_ic_init(ts);
		}
		break;
	case 2:
		if (ts->curr_pwr_state == POWER_ON) {
			touch_disable(ts);
			release_all_ts_event(ts);
			touch_power_cntl(ts, POWER_OFF);

			touch_power_cntl(ts, POWER_ON);
			msleep(ts_role->booting_delay);
			touch_enable(ts);
			touch_ic_init(ts);
		}
		break;
	default:
		TOUCH_INFO_MSG("usage: echo [0|1|2] > control\n");
		TOUCH_INFO_MSG("  - 0: power off\n");
		TOUCH_INFO_MSG("  - 1: power on\n");
		TOUCH_INFO_MSG("  - 2: power reset\n");
		break;
	}
	return count;
}
static ssize_t reg_control_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct lge_touch_data *ts = dev_get_drvdata(dev);

	int	ret;
	ret = touch_drv->sysfs(ts->client, 0, buf, SYSFS_REG_CONTROL_STORE);

	return count;
}

static ssize_t fx_control_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct lge_touch_data *ts = dev_get_drvdata(dev);

	int	ret;
	ret = touch_drv->sysfs(ts->client, 0, buf, SYSFS_FX_CONTROL_STORE);

	return count;
}


#ifdef SENSING_TEST
static ssize_t sensing_test_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct lge_touch_data *ts = dev_get_drvdata(dev);

	int	ret;
	ret = touch_drv->sysfs(ts->client, 0, buf, SYSFS_SENSING_TEST_STORE);

	return count;
}
#endif

static ssize_t chstatus_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct lge_touch_data *ts = dev_get_drvdata(dev);
	ssize_t ret;

	ret = touch_drv->sysfs(ts->client, buf, 0, SYSFS_CHSTATUS_SHOW);
	return ret;
}

static ssize_t rawdata_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct lge_touch_data *ts = dev_get_drvdata(dev);
	ssize_t ret;

	ret = touch_drv->sysfs(ts->client, buf, 0, SYSFS_RAWDATA_SHOW);
	return ret;
}

static ssize_t jitter_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct lge_touch_data *ts = dev_get_drvdata(dev);
	ssize_t ret;

	ret = touch_drv->sysfs(ts->client, buf, 0, SYSFS_JITTER_SHOW);
	return ret;
}

static ssize_t delta_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct lge_touch_data *ts = dev_get_drvdata(dev);
	ssize_t ret;

	ret = touch_drv->sysfs(ts->client, buf, 0, SYSFS_DELTA_SHOW);
	return ret;
}

static ssize_t self_diagnostic_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct lge_touch_data *ts = dev_get_drvdata(dev);
	ssize_t ret;

	ts->sd_status = 1;
	ret = touch_drv->sysfs(ts->client, buf, 0, SYSFS_SELF_DIAGNOSTIC_SHOW);
	ts->sd_status = 0;
	return ret;
}

static ssize_t self_diagnostic_status_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct lge_touch_data *ts = dev_get_drvdata(dev);
	ssize_t ret;
	ret = snprintf(buf, PAGE_SIZE, "%d\n", ts->sd_status);
	return ret;
}


static ssize_t edge_expand_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct lge_touch_data *ts = dev_get_drvdata(dev);
	ssize_t ret;

	ret = touch_drv->sysfs(ts->client, buf, 0, SYSFS_EDGE_EXPAND_SHOW);
	return ret;
}

static ssize_t edge_expand_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct lge_touch_data *ts = dev_get_drvdata(dev);
	int ret = 0;
	ret = touch_drv->sysfs(ts->client, 0, buf, SYSFS_EDGE_EXPAND_STORE);

	return count;
}

static DEVICE_ATTR(platform_data, S_IRUGO | S_IWUSR, platform_data_show, NULL);
static DEVICE_ATTR(fw_upgrade, S_IRUGO | S_IWUSR, fw_upgrade_show, fw_upgrade_store);
static DEVICE_ATTR(firmware, S_IRUGO | S_IWUSR, firmware_show, firmware_store);
static DEVICE_ATTR(version, S_IRUGO | S_IWUSR, version_show, NULL);
static DEVICE_ATTR(testmode_ver, S_IRUGO | S_IWUSR, testmode_version_show, NULL);
static DEVICE_ATTR(status, S_IRUGO | S_IWUSR, status_show, NULL);
static DEVICE_ATTR(power_control, S_IRUGO | S_IWUSR, NULL, power_control_store);
static DEVICE_ATTR(reg_control, S_IRUGO | S_IWUSR, NULL, reg_control_store);
static DEVICE_ATTR(fx_control, S_IRUGO | S_IWUSR, NULL, fx_control_store);
static DEVICE_ATTR(chstatus, S_IRUGO | S_IWUSR, chstatus_show, NULL);
static DEVICE_ATTR(rawdata, S_IRUGO | S_IWUSR, rawdata_show, NULL);
static DEVICE_ATTR(jitter, S_IRUGO | S_IWUSR, jitter_show, NULL);
static DEVICE_ATTR(delta, S_IRUGO | S_IWUSR, delta_show, NULL);
static DEVICE_ATTR(sd, S_IRUGO | S_IWUSR, self_diagnostic_show, NULL);
static DEVICE_ATTR(sd_status, S_IRUGO | S_IWUSR, self_diagnostic_status_show, NULL);
static DEVICE_ATTR(edge_expand, S_IRUGO | S_IWUSR/*S_IWUGO*/, edge_expand_show, edge_expand_store);
#ifdef SENSING_TEST
static DEVICE_ATTR(sensing_test, S_IRUGO | S_IWUSR, NULL, sensing_test_store);
#endif

static struct attribute *lge_touch_attributes[] = {
	&dev_attr_platform_data.attr,
	&dev_attr_fw_upgrade.attr,
	&dev_attr_firmware.attr,
	&dev_attr_version.attr,
	&dev_attr_testmode_ver.attr,
	&dev_attr_status.attr,
	&dev_attr_power_control.attr,
	&dev_attr_reg_control.attr,
	&dev_attr_fx_control.attr,
	&dev_attr_chstatus.attr,
	&dev_attr_rawdata.attr,
	&dev_attr_jitter.attr,
	&dev_attr_delta.attr,
	&dev_attr_sd.attr,
	&dev_attr_sd_status.attr,
	&dev_attr_edge_expand.attr,
#ifdef SENSING_TEST
	&dev_attr_sensing_test.attr,
#endif
	NULL,
};

static const struct attribute_group lge_touch_attr_group = {
	.attrs = lge_touch_attributes,
};

static int touch_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	struct lge_touch_data *ts;
	int ret = 0;

	TOUCH_TRACE_FUNC();

	if (!i2c_check_functionality(client->adapter, I2C_FUNC_I2C)) {
		TOUCH_ERR_MSG("i2c functionality check error\n");
		ret = -EPERM;
		goto err_check_functionality_failed;
	}

	ts = devm_kzalloc(&client->dev, sizeof(struct lge_touch_data), GFP_KERNEL);
	if (ts == NULL) {
		TOUCH_ERR_MSG("Can not allocate memory\n");
		ret = -ENOMEM;
		goto err_alloc_data_failed;
	}

	ts->client = client;
	i2c_set_clientdata(client, ts);
	ret = touch_init_platform_data(client);
	if (ret < 0)
		goto err_assign_platform_data;

	/* Specific device probe */
	if (touch_drv->probe) {
		ts->pdata->panel_on = 1;
		ret = touch_drv->probe(client, ts_pdata);
		if (ret < 0) {
			TOUCH_ERR_MSG("specific device probe fail\n");
			goto err_assign_platform_data;
		}
	}
#if 0

	/* reset pin setting */
	if (gpio_is_valid(ts_pdata->reset_pin)) {
		ret = gpio_request(ts_pdata->reset_pin, "touch_reset");
		if (ret < 0) {
			TOUCH_ERR_MSG("FAIL: touch_reset gpio_request\n");
			goto err_assign_platform_data;
		}
		gpio_direction_output(ts_pdata->reset_pin, 1);
	}



	/* i2c_gpio_pin */
	if (gpio_is_valid(ts_pdata->scl_pin)) {
		ret = gpio_request(ts_pdata->scl_pin, "touch_i2c_scl");
		if (ret < 0) {
			TOUCH_ERR_MSG("FAIL: touch_reset gpio_request\n");
		}
	}

	if (gpio_is_valid(ts_pdata->sda_pin)) {
		ret = gpio_request(ts_pdata->sda_pin, "touch_i2c_sda");
		if (ret < 0) {
			TOUCH_ERR_MSG("FAIL: touch_reset gpio_request\n");
		}
	}
#endif

	atomic_set(&ts->device_init, 0);

	/* Power on */
	if (touch_power_cntl(ts, POWER_ON) < 0)
		goto err_power_failed;

	msleep(ts_role->booting_delay);

	if (touch_drv->init(ts->client, &ts->fw_info) < 0) {
		TOUCH_ERR_MSG("specific device initialization fail\n");
	}

	/* init work_queue */
#if 1
	INIT_DELAYED_WORK(&ts->work_touch_lock, touch_lock_func);
#else
	if (ts_role->key_type == TOUCH_HARD_KEY) {
		INIT_WORK(&ts->work, touch_work_func_a);
		INIT_DELAYED_WORK(&ts->work_touch_lock, touch_lock_func);
	} else if (ts_role->key_type == TOUCH_SOFT_KEY) {
		INIT_WORK(&ts->work, touch_work_func_b);
	} else {
		INIT_WORK(&ts->work, touch_work_func_c);
	}
#endif

	INIT_DELAYED_WORK(&ts->work_init, touch_init_func);
	INIT_WORK(&ts->work_fw_upgrade, touch_fw_upgrade_func);
	init_completion(&ts->irq_completion);

	/* interrupt mode */
	ret = gpio_request(ts_pdata->int_pin, "touch_int");
	if (ret < 0) {
		TOUCH_ERR_MSG("FAIL: touch_int gpio_request\n");
		goto err_interrupt_failed;
	}
	gpio_direction_input(ts_pdata->int_pin);
	ret = request_threaded_irq(client->irq, touch_irq_handler,
			touch_thread_irq_handler,
			IRQF_TRIGGER_LOW | IRQF_ONESHOT, client->name, ts);

	if (ret < 0) {
		TOUCH_ERR_MSG("request_irq failed. use polling mode\n");
		goto err_interrupt_failed;
	}

#ifdef LGE_TOUCH_GHOST_DETECTION
	if (ts_role->ghost_detection_enable) {
		hrtimer_init(&ts->trigger_timer, CLOCK_MONOTONIC, HRTIMER_MODE_REL);
		ts->trigger_timer.function = touch_trigger_timer_handler;
	}
#endif
	/* ghost-finger solution */
	ts->gf_ctrl.probe = 1;

	/* Specific device initialization */
	touch_ic_init(ts);

	ret = touch_input_init(ts);
	if (ret < 0)
		goto err_lge_touch_input_init;

	ret = sysfs_create_group(&ts->input_dev->dev.kobj, &lge_touch_attr_group);

	/* Firmware Upgrade Check - use thread for booting time reduction */
	if (ts_pdata->auto_fw_update && touch_drv->fw_upgrade) {
		if (likely(touch_debug_mask_ & (DEBUG_FW_UPGRADE | DEBUG_BASE_INFO)))
			TOUCH_INFO_MSG("Auto F/W upgrade .\n");
		safety_reset(ts);
		queue_work(touch_wq, &ts->work_fw_upgrade);
	}

#if defined(CONFIG_FB)
	ts->fb_notifier_block.notifier_call = touch_notifier_callback;
	ret = fb_register_client(&ts->fb_notifier_block);
	if (ret) {
		TOUCH_ERR_MSG("unable to register fb_notifier callback: %d\n", ret);
		goto err_lge_touch_fb_register;
	}
#endif

#if defined(CONFIG_HAS_EARLYSUSPEND)
	ts->early_suspend.level = EARLY_SUSPEND_LEVEL_BLANK_SCREEN + 1;
	ts->early_suspend.suspend = touch_early_suspend;
	ts->early_suspend.resume = touch_late_resume;
	register_early_suspend(&ts->early_suspend);
#endif

#if 0
	/* Register sysfs for making fixed communication path to framework layer */
	ret = sysdev_class_register(&lge_touch_sys_class);
	if (ret < 0) {
		TOUCH_ERR_MSG("sysdev_class_register is failed\n");
		goto err_lge_touch_sys_class_register;
	}

	ret = sysdev_register(&lge_touch_sys_device);
	if (ret < 0) {
		TOUCH_ERR_MSG("sysdev_register is failed\n");
		goto err_lge_touch_sys_dev_register;
	}

	ret = kobject_init_and_add(&ts->lge_touch_kobj, &lge_touch_kobj_type,
			ts->input_dev->dev.kobj.parent,
			"%s", LGE_TOUCH_NAME);
	if (ret < 0) {
		TOUCH_ERR_MSG("kobject_init_and_add is failed\n");
		goto err_lge_touch_sysfs_init_and_add;
	}

	if (likely(touch_debug_mask_ & DEBUG_BASE_INFO))
		TOUCH_INFO_MSG("Touch driver is initialized\n");

#endif
	return 0;


#if 0
err_lge_touch_sysfs_init_and_add:
	kobject_del(&ts->lge_touch_kobj);
err_lge_touch_sys_dev_register:
	sysdev_unregister(&lge_touch_sys_device);
err_lge_touch_sys_class_register:
	sysdev_class_unregister(&lge_touch_sys_class);

#endif
#if defined(CONFIG_FB)
	fb_unregister_client(&ts->fb_notifier_block);
err_lge_touch_fb_register:
#endif
#if defined(CONFIG_HAS_EARLYSUSPEND)
	unregister_early_suspend(&ts->early_suspend);
#endif
	free_irq(client->irq, ts);

err_interrupt_failed:
err_lge_touch_input_init:
	release_all_ts_event(ts);
	touch_power_cntl(ts, POWER_OFF);
err_power_failed:
err_assign_platform_data:
err_alloc_data_failed:
err_check_functionality_failed:
	return ret;
}

static int touch_remove(struct i2c_client *client)
{
	struct lge_touch_data *ts = i2c_get_clientdata(client);

	TOUCH_TRACE_FUNC();

#if defined(CONFIG_FB)
	fb_unregister_client(&ts->fb_notifier_block);
#endif
#if defined(CONFIG_HAS_EARLYSUSPEND)
	unregister_early_suspend(&ts->early_suspend);
#endif

	/* Specific device remove */
	if (touch_drv->remove)
		touch_drv->remove(client);

	release_all_ts_event(ts);

	/* Power off */
	touch_power_cntl(ts, POWER_OFF);

	if (gpio_is_valid(ts_pdata->reset_pin)) {
		gpio_free(ts_pdata->reset_pin);
	}

	if (gpio_is_valid(ts_pdata->int_pin)) {
		gpio_free(ts_pdata->int_pin);
	}

#if 0
	kobject_del(&ts->lge_touch_kobj);
	sysdev_unregister(&lge_touch_sys_device);
	sysdev_class_unregister(&lge_touch_sys_class);
#endif

#if defined(CONFIG_HAS_EARLYSUSPEND)
	unregister_early_suspend(&ts->early_suspend);
#endif

	free_irq(client->irq, ts);

#ifdef LGE_TOUCH_GHOST_DETECTION
	if (ts_role->ghost_detection_enable) {
		hrtimer_cancel(&ts->trigger_timer);
	}
#endif
	input_unregister_device(ts->input_dev);

	return 0;
}

#if !defined(CONFIG_FB)
#if defined(CONFIG_PM)
static struct dev_pm_ops touch_pm_ops = {
#if !defined(CONFIG_HAS_EARLYSUSPEND)
	.suspend 	= touch_suspend,
	.resume 	= touch_resume,
#endif
};
#endif
#endif

static struct of_device_id touch_match_table[] = {
	{ .compatible = "lge,touch_core", },
	{ },
};

static struct i2c_device_id touch_id_table[] = {
	{ LGE_TOUCH_NAME, 0 },
};

static struct i2c_driver lge_touch_driver = {
	.probe = touch_probe,
	.remove = touch_remove,
	.id_table = touch_id_table,
	.driver = {
		.name = "lge_touch_melfas",
		.owner = THIS_MODULE,
#if !defined(CONFIG_FB)
#if defined(CONFIG_PM)
		.pm = &touch_pm_ops,
#endif
#endif
		.of_match_table = touch_match_table,
	},
};

#ifdef CONFIG_LGE_PM_CHARGING_CHARGERLOGO
extern int lge_boot_mode_for_touch;
#endif

int touch_driver_register_(struct touch_device_driver *driver)
{
	int ret = 0;

	TOUCH_TRACE_FUNC();

#ifdef CONFIG_LGE_PM_CHARGING_CHARGERLOGO
	if (lge_boot_mode_for_touch == 2) { /* Chargerlogo mode */
		return -EMLINK;
	}
#endif

	if (touch_drv != NULL) {
		TOUCH_ERR_MSG("CANNOT add new touch-driver\n");
		ret = -EMLINK;
		goto err_touch_driver_register;
	}

	touch_drv = driver;
	touch_wq = create_singlethread_workqueue("touch_wq");
	if (!touch_wq) {
		TOUCH_ERR_MSG("CANNOT create new workqueue\n");
		ret = -ENOMEM;
		goto err_work_queue;
	}

	ret = i2c_add_driver(&lge_touch_driver);
	if (ret < 0) {
		TOUCH_ERR_MSG("FAIL: i2c_add_driver\n");
		goto err_i2c_add_driver;
	}

	return 0;

err_i2c_add_driver:
	destroy_workqueue(touch_wq);
err_work_queue:
err_touch_driver_register:
	return ret;
}

void touch_driver_unregister_(void)
{
	TOUCH_TRACE_FUNC();

#ifdef CONFIG_LGE_PM_CHARGING_CHARGERLOGO
	if (lge_boot_mode_for_touch == 2) { /* Chargerlogo mode */
		return;
	}
#endif

	i2c_del_driver(&lge_touch_driver);
	touch_drv = NULL;

	if (touch_wq)
		destroy_workqueue(touch_wq);
}

