/*
 * Copyright (C)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *    * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *    * Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following disclaimer
 * in the documentation and/or other materials provided with the
 * distribution.
 *    * Neither the name of Google Inc. nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
// Magic pretend-to-be-a-chromium-build flags
#undef WEBKIT_IMPLEMENTATION
#undef LOG

#include "content/date_detector.h"

#include <bitset>

#include "base/utf_string_conversions.h"
#include "net/base/escape.h"
#include "Settings.h"
#include "WebString.h"

#include <wtf/HashSet.h>
#include <wtf/Noncopyable.h>
#include <wtf/text/StringHash.h>
#include <wtf/text/WTFString.h>

#include <android/log.h>
#include <cutils/log.h>

#include "unicode/smpdtfmt.h"
#include "unicode/ucal.h"


namespace {

std::string dateS;

#define MAX_DAY_NUM 31
#define MAX_MONTH_NUM 12
#define MIN_YEAR_NUM 1000
#define MAX_DATE_FORMATS 6

static const char kDateSchemaPrefix[] = "cal:";

// Maximum text length to be searched for date detection.
static const size_t kMaxDateLength = 24;

static const char* s_dateformats[] =
{
	"%m/%d/%Y",
	"%m-%d-%y",
	"%d-%m-%y",
	"%d/%m/%y",
	"%Y/%m/%d",
	"%Y-%m-%d",	
};

enum DATE_FORMATS {
    MM_DD_YYYY_1=0, //"mm-dd-yyyy"
	MM_DD_YYYY_2,	//"mm/dd/yyyy"
	DD_MM_YYYY_1,	//"dd-mm-yyyy"
	DD_MM_YYYY_2,	//"dd/mm/yyyy"
	YYYY_MM_DD_1,	//"yyyy-mm-dd"
	YYYY_MM_DD_2,	//"yyyy/mm/dd"
	DATE_FORMAT_MAX

};


}

DateDetector::DateDetector() {
}

DateDetector::~DateDetector() {
}

DateDetector::DateNumberParser::DateNumberParser() {
}

DateDetector::DateNumberParser::~DateNumberParser() {
}

bool DateDetector::DateNumberParser::IsPostDelimiter(
    char16 character) {
  return IsWhitespace(character) || strchr(" ", character);
}

void DateDetector::DateNumberParser::AcceptChars(size_t num_chars) {
  size_t offset = std::min(static_cast<size_t>(std::distance(it_, end_)),
      num_chars);
  it_ += offset;
  result_chars_ += offset;
}

void DateDetector::DateNumberParser::SkipChars(size_t num_chars) {
  it_ += std::min(static_cast<size_t>(std::distance(it_, end_)), num_chars);
}

void DateDetector::DateNumberParser::ResetState() {
  num_digits_ = 0;
  result_chars_ = 0;
}

bool DateDetector::DateNumberParser::CheckFinished(Word* word) const {
  // There should always be a number after a hyphen.
  if (result_chars_ == 0)
    return false;

  if (word) {
    word->begin = it_ - result_chars_;
    word->end = it_;
	
	std::string sstart( word->begin, word->end );	
	
	int i = 0, month = 0,day = 0,year = 0;
	char delimiter;
	std::istringstream is(sstart);
    struct tm ttm;
    
	for(i=0; i < DATE_FORMAT_MAX; i++)
	{
    	if (strptime(sstart.c_str(), s_dateformats[i], &ttm))
	    {
	    	switch(i)
			{
			case MM_DD_YYYY_1:			
			case MM_DD_YYYY_2:
				is >> month >> delimiter >> day >> delimiter >> year;
				break;
			case DD_MM_YYYY_1:
			case DD_MM_YYYY_2:
				is >> day >> delimiter >> month >> delimiter >> year;
				break;
			case YYYY_MM_DD_1:
			case YYYY_MM_DD_2:
				is >> year >> delimiter >> month >> delimiter >> day;
				break;
			}
			std::stringstream ss;
			if(day && month && year &&
				(year < 100 || year > 1000 ) &&
				(day < (MAX_DAY_NUM + 1)) &&
				(month < (MAX_MONTH_NUM + 1)))
			{
				if(year < 100)
				{
					year += 2000;
				}
				year = std::min(year, 2036);	//max year is 2036
				ss << day << "-" << month << "-" << year;
				dateS =ss.str();
				return true;
			}
			else
				return false;
    	}
	}
  }
  
  return false;
}

bool DateDetector::DateNumberParser::Parse(
    const string16::const_iterator& begin,
    const string16::const_iterator& end, Word* word) {

  it_ = begin;
  end_ = end;
  ResetState();

  // Iterations only used as a fail-safe against any buggy infinite loops.
  size_t iterations = 0;
  size_t max_iterations = end - begin + 1;
  
  for (; it_ != end_ && iterations < max_iterations; ++iterations) {
    // Word finished case.
    if (IsPostDelimiter(*it_)) {
      if (CheckFinished(word))
        return true;
      else if (result_chars_)
        ResetState();

      SkipChars(1);
      continue;
    }
	
    // More digits. There should be no more after a letter was found.
    if (IsAsciiDigit(*it_) || (*it_ == '-' || *it_ == '/')) {
      AcceptChars(1);
      continue;
    }
	
    SkipChars(1);
  }

  if (iterations >= max_iterations)
    return false;

  return false;
}


bool DateDetector::FindContent(const string16::const_iterator& begin,
    const string16::const_iterator& end, size_t* start_pos, size_t* end_pos) {

	DateNumberParser date_number_parser;
	Word number;
	// Keep going through the input string until a potential date number is
	// detected. Start tokenizing the following words to find a valid
	// month within a word range  followed by a valid year code. 
	if (date_number_parser.Parse(begin, end, &number))
	{
		*start_pos = number.begin - begin;
		*end_pos = number.end - begin;
		return true;
	}
	return false;
}

std::string DateDetector::GetContentText(const WebKit::WebRange& range) {
       return UTF16ToUTF8(range.toPlainText());
}

GURL DateDetector::GetIntentURL(const std::string& content_text) {
__android_log_print(ANDROID_LOG_INFO, "H_DATE_LOGS", "GetIntentURL Date string %s",dateS.c_str());
	return GURL(kDateSchemaPrefix + EscapeQueryParamValue(dateS, true));
}

size_t DateDetector::GetMaximumContentLength() {
	return kMaxDateLength;
}

bool DateDetector::IsEnabled(const WebKit::WebHitTestInfo& hit_test) {
	WebCore::Settings* settings = GetSettings(hit_test);
	return settings && settings->formatDetectionDate();
}

